package ru.yandex.webmaster3.worker.checklist.badad;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.checklist.data.YaBrowserBadAdFormat;
import ru.yandex.webmaster3.core.checklist.data.YaBrowserBadAdSample;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.AllHostsCacheService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.util.JsonDBMapping;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtOperationId;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.host.CommonDataType.LAST_YABROWSER_BADADS_UPDATE;

/**
 * Created by Oleg Bazdyrev on 30/11/2017.
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Service
public class UpdateYaBrowserBadAdsTask extends PeriodicTask<PeriodicTaskState> {
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new ParameterNamesModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    private final AllHostsCacheService allHostsCacheService;
    private final CommonDataStateYDao commonDataStateYDao;
    private final SiteProblemsService siteProblemsService;
    private final ChecklistPageSamplesService checklistPageSamplesService;

    private final YtService ytService;
    @Value("${webmaster3.worker.badad.sourceTable.path}")
    private YtPath sourceTable;
    @Value("${webmaster3.worker.badad.tmpSortedTable.path}")
    private YtPath tmpSortedTable;

    @Override
    public Result run(UUID runId) throws Exception {
        DateTime updateStarted = DateTime.now();
        CommonDataState lastUpdate = commonDataStateYDao.getValue(LAST_YABROWSER_BADADS_UPDATE);
        MutableObject<DateTime> nodeModificationTime = new MutableObject<>();
        ytService.inTransaction(sourceTable).execute(cypressService -> {
            YtNode node = cypressService.getNode(sourceTable);
            if (lastUpdate == null || new DateTime(Long.parseLong(lastUpdate.getValue())).isBefore(node.getUpdateTime())) {
                log.info("New data found. Updating bad ads from table with update time {}", node.getUpdateTime());
                nodeModificationTime.setValue(node.getUpdateTime());
                updateBadAds(cypressService);
            } else {
                log.info("No new data found");
            }
            return true;
        });

        if (nodeModificationTime.getValue() != null) {
            siteProblemsService.notifyCleanableProblemUpdateFinished(SiteProblemTypeEnum.YABROWSER_BADAD, updateStarted);
            commonDataStateYDao.update(new CommonDataState(LAST_YABROWSER_BADADS_UPDATE,
                    String.valueOf(nodeModificationTime.getValue().getMillis()), DateTime.now()));
        }

        return new Result(TaskResult.SUCCESS);
    }

    private void updateBadAds(YtCypressService cypressService) throws YtException {
        log.info("Sorting source table by host");
        // отсортируем исходную таблицу
        if (cypressService.exists(tmpSortedTable)) {
            cypressService.remove(tmpSortedTable);
        }
        YtOperationId operationId = cypressService.sort(sourceTable, tmpSortedTable, "host");
        if (!cypressService.waitFor(operationId)) {
            throw new WebmasterException("Could not wait for sort operation completion",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), null));
        }
        // читаем отсортированную
        AsyncTableReader<YaBrowserBadAdRow> tableReader = new AsyncTableReader<>(cypressService, tmpSortedTable,
                Range.all(), YtTableReadDriver.createYSONDriver(YaBrowserBadAdRow.class, OM))
                .withRetry(3).splitInParts(1000L);

        try (AsyncTableReader.TableIterator<YaBrowserBadAdRow> iterator = tableReader.read()) {
            WebmasterHostId prevHostId = null;
            DateTime lastUpdate = null;
            DateTime warnExpirationDate = null;
            SiteProblemContent.YaBrowserBadAd.YaBrowserBadAdStatus status = null;
            Set<String> samples = new HashSet<>();
            EnumSet<YaBrowserBadAdFormat> badAdFormats = EnumSet.noneOf(YaBrowserBadAdFormat.class);
            while (iterator.hasNext()) {
                YaBrowserBadAdRow row = iterator.next();
                if (CollectionUtils.isEmpty(row.getViolations())) {
                    log.warn("Host {} has no data", row.getHost());
                    continue;
                }
                // хост возьмем из примера
                WebmasterHostId hostId = IdUtils.urlToHostId(row.getViolations().get(0).getExamples().get(0));
                // сохраним накопившиеся данные, если они есть
                if (prevHostId != null && !hostId.equals(prevHostId)) {
                    saveHostData(prevHostId, badAdFormats, samples, lastUpdate, warnExpirationDate, status);
                    samples = new HashSet<>();
                    badAdFormats = EnumSet.noneOf(YaBrowserBadAdFormat.class);
                    lastUpdate = null;
                    warnExpirationDate = null;
                }
                // проверим наличие хоста в вебмастере
                if (!allHostsCacheService.contains(hostId)) {
                    prevHostId = null;
                    continue;
                }
                // сохраняем все примеры и плохие форматы в одну кучу
                status = SiteProblemContent.YaBrowserBadAd.YaBrowserBadAdStatus.valueOf(row.getStatus());
                warnExpirationDate = ObjectUtils.min(warnExpirationDate, secondsToDateTime(row.getWarnExpirationTs()));
                lastUpdate = ObjectUtils.max(lastUpdate, secondsToDateTime(row.getDetectionTs()));
                // форматы
                for (Violation violation : row.getViolations()) {
                    YaBrowserBadAdFormat adFormat = YaBrowserBadAdFormat.fromRawId(violation.getFormat());
                    if (adFormat == null) {
                        log.error("Unknown bad ad format code {}", violation.getFormat());
                        continue;
                    }
                    badAdFormats.add(adFormat);
                    // сохраняем примеры
                    for (String example : violation.getExamples()) {
                        YaBrowserBadAdSample sample = new YaBrowserBadAdSample(adFormat, example);
                        samples.add(JsonDBMapping.OM.writeValueAsString(sample));
                    }
                }
                prevHostId = hostId;
            }
            // сохраним последний хост
            if (prevHostId != null) {
                saveHostData(prevHostId, badAdFormats, samples, lastUpdate, warnExpirationDate, status);
            }

        } catch (IOException | InterruptedException e) {
            throw new WebmasterException("Yt error", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }

    }

    @Nullable
    private DateTime secondsToDateTime(Long seconds) {
        return seconds == null || seconds == 0L ? null : new DateTime(TimeUnit.SECONDS.toMillis(seconds));
    }

    private void saveHostData(WebmasterHostId hostId, EnumSet<YaBrowserBadAdFormat> badAdFormats,
                              Collection<String> urls, DateTime lastUpdate, DateTime warnExpirationDate,
                              SiteProblemContent.YaBrowserBadAd.YaBrowserBadAdStatus status) {
        if (badAdFormats.isEmpty()) {
            return;
        }
        ProblemSignal problemSignal = new ProblemSignal(new SiteProblemContent.YaBrowserBadAd(lastUpdate, null,
                warnExpirationDate, status, badAdFormats), lastUpdate);
        // save problem
        siteProblemsService.updateCleanableProblem(hostId, problemSignal);
        // save samples
        checklistPageSamplesService.saveSamples(hostId, ChecklistSamplesType.YABROWSER_BADAD, new ArrayList<>(urls));
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPDATE_YABROWSER_BADADS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("5 5 0/3 * * *");
    }

    public static final class YaBrowserBadAdRow {

        private final String host;
        private final String platform;
        private final String status;
        private final Long detectionTs;
        private final Long warnExpirationTs;
        private final List<Violation> violations;

        public YaBrowserBadAdRow(String host, String platform, String status, Long detectionTs, Long warnExpirationTs,
                                 List<Violation> violations) {
            this.host = host;
            this.platform = platform;
            this.status = status;
            this.detectionTs = detectionTs;
            this.warnExpirationTs = warnExpirationTs;
            this.violations = violations;
        }

        public String getHost() {
            return host;
        }

        public String getPlatform() {
            return platform;
        }

        public String getStatus() {
            return status;
        }

        @JsonProperty("detection_ts")
        public Long getDetectionTs() {
            return detectionTs;
        }

        @JsonProperty("warn_expiration_ts")
        public Long getWarnExpirationTs() {
            return warnExpirationTs;
        }

        public List<Violation> getViolations() {
            return violations;
        }
    }

    public static final class Violation {

        private final String format;
        private final List<String> examples;

        public Violation(String format, List<String> examples) {
            this.format = format;
            this.examples = examples;
        }

        public String getFormat() {
            return format;
        }

        public List<String> getExamples() {
            return examples;
        }
    }

    /** Пример структуры
     * [
     {
     "examples": [
     "http://volozh.blog/1",
     "http://volozh.blog/2",
     "http://volozh.blog/3"
     ],
     "format": "desktop-audio-auto"
     },
     {
     "examples": [
     "http://volozh.blog/4",
     "http://volozh.blog/5",
     "http://volozh.blog/6"
     ],
     "format": "desktop-fullscr-onload-count"
     }
     ]
     */

}
