package ru.yandex.webmaster3.worker.clickhouse;

import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.dao.ClickhouseTablesRepository;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.clickhouse.system.data.ClickhouseSystemTableInfo;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.clickhouse.TableType.ACHIEVEMENTS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.ACHIEVEMENTS_KUUB;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.ALL_MIRRORS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.CURRENT_SERP_DATA;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.DISPLAY_NAME;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.DOMAINS_ON_SEARCH;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.EXTENDED_RECOMMENDED_QUERIES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.EXTERNAL_DELETED_LINK_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.EXTERNAL_LINK_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.GOODS_OFFERS_LOGS_HISTORY;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.GREEN_URLS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOSTS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOST_FAVICONS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOST_PROBLEMS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOST_REGIONS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOST_STATISTICS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.HOST_THREATS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.IKS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.IMPORTANT_URLS_HISTORY;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.IMPORTANT_URLS_LAST_STATE;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.INTERNAL_LINK_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.LAST_SITE_STRUCTURE;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.MEMORANDUM_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.METRIKA_CRAWL_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.METRIKA_STATS_BY_TIME;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.MIRRORS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.MIRRORS2;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.OWNER_THREATS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.RECOMMENDED_QUERIES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.RECOMMENDED_URLS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.RIVALS_STATS2;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.SITEMAPS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.SITEMAP_PROBLEMS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.SITE_SERVICES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TOP_3000__QUERIES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TOP_3000__VALUES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TOP_URLS_TEXTS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TOP_URLS_VALUES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_AUTO_DELETED_PAGES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_DOMAINS_STATE;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_FEEDS_HISTORY;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_PROBLEMS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_SEARCHURLS_SAMPLES;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.TURBO_SEARCHURLS_STATS;
import static ru.yandex.webmaster3.storage.clickhouse.TableType.WEEK_QUERIES_SMALL;

/**
 * Периодическая таска, чистящая старые ненужные таблицы
 * Created by Oleg Bazdyrev on 18/05/2017.
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class CleanClickhouseTablesTask extends PeriodicTask<PeriodicTaskState> {

    private static final Logger log = LoggerFactory.getLogger(CleanClickhouseTablesTask.class);

    private static final Duration MIN_AGE_FOR_SINGLE_TABLES = Duration.standardHours(12L);
    private static final String DISTRIB_SUFFIX = "_distrib";
    private static final Map<TableType, CleanConfig> TABLE_TYPES = ImmutableMap.<TableType, CleanConfig>builder()
            .put(WEEK_QUERIES_SMALL, createCleanConfig(1))
            .put(TOP_3000__QUERIES, createCleanConfig(1))
            .put(TOP_3000__VALUES, createCleanConfig(1))
            .put(TOP_URLS_TEXTS, createCleanConfig(1))
            .put(TOP_URLS_VALUES, createCleanConfig(1))
            .put(INTERNAL_LINK_SAMPLES, createCleanConfig(2))
            .put(EXTERNAL_LINK_SAMPLES, createCleanConfig(2))
            .put(EXTERNAL_DELETED_LINK_SAMPLES, createCleanConfig(2))
            .put(HOST_THREATS, createCleanConfig(2))
            .put(RECOMMENDED_QUERIES, createCleanConfig(2))
            .put(EXTENDED_RECOMMENDED_QUERIES, createCleanConfig(2))
            .put(RIVALS_STATS2, createCleanConfig(2))
            .put(MEMORANDUM_SAMPLES, createCleanConfig(2))
            .put(ACHIEVEMENTS, createCleanConfig(5))
            .put(ACHIEVEMENTS_KUUB, createCleanConfig(5))
            .put(SITE_SERVICES, createCleanConfig(5))
            .put(TURBO_SEARCHURLS_STATS, createCleanConfig(2))
            .put(TURBO_SEARCHURLS_SAMPLES, createCleanConfig(2))
            .put(IMPORTANT_URLS_HISTORY, createCleanConfig(2))
            .put(IMPORTANT_URLS_LAST_STATE, createCleanConfig(2))
            .put(METRIKA_CRAWL_SAMPLES, createCleanConfig(5))
            .put(OWNER_THREATS, createCleanConfig(5))
            .put(MIRRORS, createCleanConfig(2, EnumSet.of(TableState.ON_LINE), EnumSet.of(TableState.ARCHIVED)))
            .put(DISPLAY_NAME, createCleanConfig(2, EnumSet.of(TableState.ON_LINE), EnumSet.of(TableState.ARCHIVED)))
            .put(HOST_FAVICONS, createCleanConfig(2))
            .put(TURBO_DOMAINS_STATE, createCleanConfig(1))
            .put(TURBO_PROBLEMS, createCleanConfig(1))
            .put(TURBO_FEEDS_HISTORY, createCleanConfig(1))
            .put(TURBO_AUTO_DELETED_PAGES, createCleanConfig(2))
            .put(HOST_STATISTICS, createCleanConfig(10))
            .put(HOST_PROBLEMS, createCleanConfig(10))
            .put(DOMAINS_ON_SEARCH, createCleanConfig(2))
            .put(MIRRORS2, createCleanConfig(2, EnumSet.of(TableState.ON_LINE), EnumSet.of(TableState.ARCHIVED)))
            .put(ALL_MIRRORS, createCleanConfig(3))
            .put(HOSTS, createCleanConfig(5))
            .put(IKS, createCleanConfig(3))
            .put(HOST_REGIONS, createCleanConfig(10))
            .put(SITEMAPS, createCleanConfig(5))
            .put(SITEMAP_PROBLEMS, createCleanConfig(5))
            .put(LAST_SITE_STRUCTURE, createCleanConfig(5))
            .put(CURRENT_SERP_DATA, createCleanConfig(5))
            .put(METRIKA_STATS_BY_TIME, createCleanConfig(5))
            .put(RECOMMENDED_URLS, createCleanConfig(5))
            .put(GREEN_URLS, createCleanConfig(1))
            .put(GOODS_OFFERS_LOGS_HISTORY, createCleanConfig(5))
            .build();

    @Setter
    private ClickhouseTablesRepository clickhouseTablesCDao;
    @Setter
    private ClickhouseServer clickhouseServer;
    @Setter
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;
    @Setter
    private PeriodicTaskType type;

    @Override
    public Result run(UUID runId) throws Exception {

        cleanupOldLinksTables();
        deleteOldTables();
        return new Result(TaskResult.SUCCESS);
    }

    @Description("Удаляет все таблицы у которых статус задан в statesForDelete")
    private void deleteOldTables() {
         clickhouseTablesCDao.listTables().stream()
                .filter(table -> TABLE_TYPES.containsKey(table.getType()))
                .filter(table -> TABLE_TYPES.get(table.getType()).getStatesForDelete().contains(table.getState()))
                .forEach(tableInfo -> {
                    removeClickhouseTables(tableInfo);
                    clickhouseTablesCDao.delete(tableInfo);
                });
    }

    @Description("Чистит таблицы с заданным статусом в statesForClean, до заданного кол-ва")
    private void cleanupOldLinksTables() {
        // для каждого из наших типов оставим только saveOldTablesCount последних онлайн табличек. а все остальные удалим
        Map<TableType, List<ClickhouseTableInfo>> tablesByType = clickhouseTablesCDao.listTables().stream()
                .filter(table -> TABLE_TYPES.containsKey(table.getType()))
                .filter(table -> TABLE_TYPES.get(table.getType()).getStatesForClean().contains(table.getState()))
                .sorted(Comparator.comparing(ClickhouseTableInfo::getUpdateDate)
                        .thenComparing(ClickhouseTableInfo::getUnixTimestamp))
                .collect(Collectors.groupingBy(ClickhouseTableInfo::getType));
        DateTime minUpdateDateForRemoveSingleTable = DateTime.now().minus(MIN_AGE_FOR_SINGLE_TABLES);
        // проходим отдельно по каждому типу
        for (Map.Entry<TableType, List<ClickhouseTableInfo>> entry : tablesByType.entrySet()) {
            List<ClickhouseTableInfo> tableInfos = entry.getValue();
            final int saveOldTablesCount = TABLE_TYPES.get(entry.getKey()).getSaveOldTablesCount();
            while (tableInfos.size() > saveOldTablesCount) {
                // если оставляем только одну табличку - то после переключения на новую должно перейти не менее X часов
                if (saveOldTablesCount == 1 && tableInfos.size() == 2) {
                    if (Iterables.getLast(tableInfos).getUpdateDate().isAfter(minUpdateDateForRemoveSingleTable)) {
                        // рано удалять
                        break;
                    }
                }
                // хитрая логика, может быть несколько табличек с одним именем, реально будем удалять из
                // кликхауса, если это последняя таблица
                ClickhouseTableInfo tableInfo = tableInfos.get(0);
                boolean removeChTables = tableInfos.stream().filter(
                        cti -> cti.getClickhouseFullName().equals(tableInfo.getClickhouseFullName())).count() == 1;
                if (removeChTables) {
                    removeClickhouseTables(tableInfo);
                }
                // все снесли, можно удалить запись из Кассандры
                clickhouseTablesCDao.delete(tableInfo);
                tableInfos.remove(0);
            }
        }
    }

    private void removeClickhouseTables(ClickhouseTableInfo tableInfo) {
        log.info("Cleaning up table: " + tableInfo.getClickhouseFullName());
        String[] parts = tableInfo.getClickhouseFullName().split("\\.");
        if (parts.length != 2) {
            log.warn("Unsupported name of table: " + tableInfo.getClickhouseFullName());
            return;
        }
        String database = parts[0];
        String namePrefix = parts[1];
        if (namePrefix.endsWith(DISTRIB_SUFFIX)) {
            namePrefix = namePrefix.substring(0, namePrefix.length() - DISTRIB_SUFFIX.length());
        }
        // проходимся по каждому хосту
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext
                    .useDefaults()
                    .setTimeout(Duration.standardMinutes(10))
                    .setHost(host);
            for (ClickhouseSystemTableInfo table : clickhouseSystemTablesCHDao.getTablesForPrefix(host, database, namePrefix)) {
                // удаляем табличку
                clickhouseServer.execute(ctx,
                        ClickhouseServer.QueryType.INSERT, "DROP TABLE IF EXISTS " + table.getDatabase() + "." + table.getName(),
                        Optional.empty(), Optional.empty());
            }
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return type;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 28/30 * * * *");
    }

    @Value
    private static class CleanConfig {
        @Description("кол-во таблиц, которые не удалятся")
        int saveOldTablesCount;
        @Description("статусы записей, при которых таблицы будут удаляться до определенного кол-ва")
        EnumSet<TableState> statesForClean;

        @Description("статусы записей, при которых таблицы будут удаляться полностью")
        EnumSet<TableState> statesForDelete;
    }

    private static CleanConfig createCleanConfig(int saveOldTablesCount) {
        return createCleanConfig(saveOldTablesCount, EnumSet.of(TableState.ON_LINE, TableState.PRE_ONLINE), EnumSet.noneOf(TableState.class));
    }

    private static CleanConfig createCleanConfig(int saveOldTablesCount, EnumSet<TableState> statesForClean, EnumSet<TableState> statesForDelete) {
        return new CleanConfig(saveOldTablesCount, statesForClean, statesForDelete);
    }

}
