package ru.yandex.webmaster3.worker.crawl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.crawl.dao.RotorSettingsYDao;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.storage.util.yt.YtUtils;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.UUID;

/**
 * @author leonidrom
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component
public class ExportRotorSettingsPeriodicTask  extends PeriodicTask<ExportRotorSettingsPeriodicTask.TaskState> {
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));

    private static final String TABLE_NAME = "rotor-settings";
    private static final String TABLE_SCHEMA = "[" +
            "{'name': 'Host', 'type': 'string'}, " +
            "{'name': 'RenderSettings', 'type': 'string'}]";

    @Value("${external.yt.service.arnold.root.default}/export")
    private final YtPath workDir;
    private final YtService ytService;

    private final RotorSettingsYDao rotorSettingsYDao;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new TaskState());

        YtPath tablePath = YtPath.path(workDir, TABLE_NAME);
        YtTableData table = null;

        try {
            table = ytService.prepareTableData(tablePath.getName(), tw -> {
                rotorSettingsYDao.forEach(rotorSettings -> {
                    String host = IdUtils.toHostString(rotorSettings.hostId(), true, false, false);
                    tw.column(F.HOST, host);
                    tw.column(F.RENDER_SETTINGS, rotorSettings.renderSettings().toString());
                    tw.rowEnd();
                    state.totalHosts++;
                });
            });

            var tp = new YtUtils.TransactionWriterBuilder(tablePath, table)
                    .withSchema(TABLE_SCHEMA)
                    .withRetry(RETRY_POLICY)
                    .build();
            var writer = new YtUtils.TransactionExecutor(ytService, workDir);
            writer.execute(tp);
        } finally {
            if (table != null) {
                table.delete();
            }
        }

        return Result.SUCCESS;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_ROTOR_SETTINGS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 */30 * * * *");
    }

    private static class F {
        static final String HOST = "Host";
        static final String RENDER_SETTINGS = "RenderSettings";
    }

    public static class TaskState implements PeriodicTaskState {
        public int totalHosts;
    }
}
