package ru.yandex.webmaster3.worker.crawl;

import java.util.UUID;

import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.crawl.dao.UserCrawlSettingsYDao;
import ru.yandex.webmaster3.storage.crawl.service.CrawlSettingsService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.storage.util.yt.YtTransactionService;
import ru.yandex.webmaster3.storage.util.yt.YtUtils;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 24.01.18.
 */
public class ExportUserCrawlSettingsPeriodicTask extends PeriodicTask<ExportUserCrawlSettingsPeriodicTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(ExportUserCrawlSettingsPeriodicTask.class);

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));

    private static final String TABLE_NAME = "crawl-settings";
    private static final String TABLE_SCHEMA = "[" +
            "{'name': 'host_url', 'type': 'string'}, " +
            "{'name': 'use_defaults', 'type': 'boolean'}, " +
            "{'name': 'multiplier', 'type': 'double'}]";

    private YtPath workDir;
    private YtService ytService;

    private UserCrawlSettingsYDao userCrawlSettingsYDao;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_USER_CRAWL_SETTINGS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 21 * * *");
    }

    @Override
    public PeriodicTask.Result run(UUID runId) throws Exception {
        state = new ExportUserCrawlSettingsPeriodicTask.TaskState();
        setState(state);

        YtPath tablePath = YtPath.path(workDir, TABLE_NAME);
        YtTableData table = null;
        try {
            table = ytService.prepareTableData(tablePath.getName(), (tw) -> {
                try {
                    userCrawlSettingsYDao.forEach(pair -> {
                        String host = IdUtils.toHostString(pair.getLeft(), true, false, false);
                        tw.column(YtRow.F_HOST_URL, host);
                        tw.columnObject(YtRow.F_USE_DEFAULTS, pair.getRight().isUseDefaults());
                        double multiplier = pair.getRight().getMultiplier();
                        if (pair.getRight().isUseDefaults()) {
                            multiplier = CrawlSettingsService.DEFAULT_USER_CRAWL_SETTINGS.getMultiplier();
                        }
                        tw.columnObject(YtRow.F_MULTIPLIER, multiplier);

                        try {
                            tw.rowEnd();
                        } catch (YtException e) {
                            throw new RuntimeException(e);
                        }
                    });
                } catch (WebmasterYdbException e) {
                    throw new RuntimeException(e);
                }
            });

            YtTransactionService.TransactionProcess process = new YtUtils.TransactionWriterBuilder(tablePath, table)
                    .withSchema(TABLE_SCHEMA)
                    .withRetry(RETRY_POLICY)
                    .build();

            YtUtils.TransactionExecutor writer = new YtUtils.TransactionExecutor(ytService, workDir);
            writer.execute(process);
        } finally {
            if (table != null) {
                table.delete();
            }
        }
        return new PeriodicTask.Result(TaskResult.SUCCESS);
    }

    @Required
    public void setWorkDir(YtPath workDir) {
        this.workDir = workDir;
    }

    @Required
    public void setYtService(YtService ytService) {
        this.ytService = ytService;
    }

    @Required
    public void setUserCrawlSettingsYDao(UserCrawlSettingsYDao userCrawlSettingsYDao) {
        this.userCrawlSettingsYDao = userCrawlSettingsYDao;
    }

    public static class TaskState implements PeriodicTaskState {
    }

    private static class YtRow {
        static final String F_HOST_URL = "host_url";
        static final String F_USE_DEFAULTS = "use_defaults";
        static final String F_MULTIPLIER = "multiplier";
    }
}
