package ru.yandex.webmaster3.worker.crawl;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.mutable.MutableDouble;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.crawl.BaseCrawlInfo;
import ru.yandex.webmaster3.storage.crawl.dao.BaseCrawlInfoYDao;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtMissingValueMode;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtRowMapper;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 23.01.18.
 */
@Slf4j
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UploadBaseCrawlInfoPeriodicTask extends PeriodicTask<UploadBaseCrawlInfoPeriodicTask.TaskState> {
    public static final String DEFAULT_RPS_ATTRIBUTE_NAME = "default_rps";

    private final BaseCrawlInfoYDao baseCrawlInfoYDao;
    private final SettingsService settingsService;
    private final YtService ytService;
    @Value("${webmaster3.worker.crawldelay.basecrawl.path}")
    private YtPath ytPath;

    @Override
    public Result run(UUID runId) {
        setState(new UploadBaseCrawlInfoPeriodicTask.TaskState());
        Stopwatch stopwatch = Stopwatch.createStarted();
        DateTime now = DateTime.now();

        Double defaultRps = getDefaultRpsAttribute(ytPath);
        if (!defaultRps.isNaN()) {
            settingsService.update(CommonDataType.CRAWL_DELAY_DEFAULT_RPS, String.valueOf(defaultRps));
        }
        MutableInt count = new MutableInt(0);
        processYtResults(rows -> {
            List<Pair<String, BaseCrawlInfo>> info = rows.stream().map(row ->
                    Pair.of(row.getHostPattern().toLowerCase(), new BaseCrawlInfo(row.getRps()))
            ).collect(Collectors.toList());
            try {
                RetryUtils.execute(RetryUtils.linearBackoff(5, Duration.standardSeconds(2)), () -> {
                    baseCrawlInfoYDao.updateInfo(info, now);
                    count.setValue(count.getValue() + info.size());
                    log.info("In total {} rows downloaded to ydb", count.getValue());
                });
            } catch (InterruptedException e) {
                throw new WebmasterException("Failed save to ydb",
                        new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
            }
        });

        settingsService.update(CommonDataType.CRAWL_DELAY_IMPORT_DATE, String.valueOf(now));
        getState().workTimeMs = stopwatch.elapsed(TimeUnit.MILLISECONDS);
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_BASE_CRAWL_INFO;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 21 * * *");
    }

    private Double getDefaultRpsAttribute(YtPath path) {
        MutableDouble result = new MutableDouble(Double.NaN);
        try {
            ytService.withoutTransaction(cypressService -> {
                YtNode node = cypressService.getNode(path);
                JsonNode attribute = node.getNodeMeta().get(DEFAULT_RPS_ATTRIBUTE_NAME);
                if (attribute != null) {
                    result.setValue(attribute.doubleValue());
                }
                return true;
            });
        } catch (YtException | InterruptedException e) {
            throw new WebmasterException("Failed to read YT results table",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }
        return result.getValue();
    }

    public void processYtResults(Consumer<List<UploadBaseCrawlInfoPeriodicTask.YtResultRow>> rowConsumer) {
        ExecutorService executorService = Executors.newFixedThreadPool(1);
        try {
            processYtResultTable(executorService, ytPath, rowConsumer);
        } catch (WebmasterException e) {
            throw new RuntimeException(e);
        }

        executorService.shutdown();
        try {
            executorService.awaitTermination(5, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            log.warn("Unable to terminate executor", e);
        }
    }

    private void processYtResultTable(ExecutorService executorService, YtPath tablePath, Consumer<List<UploadBaseCrawlInfoPeriodicTask.YtResultRow>> rowConsumer) {
        try {
            ytService.withoutTransaction(cypressService -> {
                readYtResultTable(executorService, cypressService, tablePath, rowConsumer);
                return true;
            });
        } catch (YtException | InterruptedException e) {
            throw new WebmasterException("Failed to read YT results table",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }
    }

    private void readYtResultTable(ExecutorService executorService, YtCypressService cypressService,
                                   YtPath tablePath, Consumer<List<UploadBaseCrawlInfoPeriodicTask.YtResultRow>> rowConsumer) throws YtException {
        AsyncTableReader<UploadBaseCrawlInfoPeriodicTask.YtResultRow> tableReader =
                new AsyncTableReader<>(cypressService, tablePath, Range.all(), new UploadBaseCrawlInfoPeriodicTask.YtResultRowMapper(),
                        YtMissingValueMode.PRINT_SENTINEL)
                        .splitInParts(200_000)
                        .inExecutor(executorService, "base-crawl-cacher")
                        .withRetry(5);

        log.trace("readYtResultTable: {}", tablePath);
        MutableInt count = new MutableInt(0);
        List<UploadBaseCrawlInfoPeriodicTask.YtResultRow> rows = new ArrayList<>();
        try (AsyncTableReader.TableIterator<UploadBaseCrawlInfoPeriodicTask.YtResultRow> it = tableReader.read()) {
            while (it.hasNext()) {
                UploadBaseCrawlInfoPeriodicTask.YtResultRow row = it.next();
                rows.add(row);
                count.increment();
                if (rows.size() == 5_000) {
                    log.info("{} rows read from yt", count.getValue());
                    rowConsumer.accept(rows);
                    rows.clear();
                }
            }
        } catch (IOException | InterruptedException e) {
            throw new YtException("Unable to read table: " + tablePath, e);
        }
        if (rows.size() > 0) {
            rowConsumer.accept(rows);
        }
    }

    private static class YtResultRow {
        static final String HOST_PATTERN = "host_url";
        static final String RPS = "rps";

        Map<String, String> fields = new HashMap<>();

        private String getHostPattern() {
            return getString(HOST_PATTERN);
        }

        private double getRps() {
            return getDouble(RPS);
        }

        private String getString(String field) {
            return fields.get(field);
        }

        private double getDouble(String field) {
            return Double.valueOf(fields.get(field));
        }
    }

    private static class YtResultRowMapper implements YtRowMapper<UploadBaseCrawlInfoPeriodicTask.YtResultRow> {
        private UploadBaseCrawlInfoPeriodicTask.YtResultRow row;

        YtResultRowMapper() {
            this.row = new UploadBaseCrawlInfoPeriodicTask.YtResultRow();
        }

        @Override
        public void nextField(String name, InputStream data) {
            try {
                String fieldStr = IOUtils.toString(data, StandardCharsets.UTF_8);
                row.fields.put(name, fieldStr);
            } catch (Exception e) {
                log.error("Unable to read Yt result field: {}", name, e);
            }
        }

        @Override
        public UploadBaseCrawlInfoPeriodicTask.YtResultRow rowEnd() {
            UploadBaseCrawlInfoPeriodicTask.YtResultRow r = row;
            log.trace("{}", row.fields.toString());
            row = new UploadBaseCrawlInfoPeriodicTask.YtResultRow();

            return r;
        }

        @Override
        public List<String> getColumns() {
            return Arrays.asList(YtResultRow.HOST_PATTERN, YtResultRow.RPS);
        }
    }

    public static class TaskState implements PeriodicTaskState {
        long workTimeMs;

        public long getWorkTimeMs() {
            return workTimeMs;
        }
    }
}

