package ru.yandex.webmaster3.worker.dashboard;

import java.util.List;
import java.util.Optional;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableProvider;
import ru.yandex.webmaster3.storage.clickhouse.dao.LegacyClickhouseTablesYDao;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.clickhouse.system.data.ClickhouseSystemTableInfo;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

@Component("/tableInfo/delete")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Slf4j
public class DeleteTableAction extends Action<DeleteTableAction.Request, DeleteTableAction.Response> {

    private final ClickhouseServer legacyMdbClickhouseServer;
    private final ClickhouseSystemTablesCHDao legacyMdbClickhouseSystemTablesCHDao;
    private final LegacyClickhouseTablesYDao legacyClickhouseTablesYDao;
    private final TableProvider tableStorage;
    private static final String DISTRIB_SUFFIX = "_distrib";

    @Override
    public Response process(Request request) throws WebmasterException {
        final ClickhouseTableInfo tableForDelete = request.getNewTableInfo();
        final List<ClickhouseTableInfo> tableInfos = legacyClickhouseTablesYDao.listTables(tableForDelete.getType());

        final ClickhouseTableInfo tableInfo = tableInfos.stream().filter(tmp -> tmp.getTableId().equals(tableForDelete.getTableId())).findFirst().orElseThrow(IllegalStateException::new);
        if (tableInfo.getState().isOnline()) {
            throw new IllegalStateException("table must be not online");
        }
        if (tableInfos.stream().anyMatch(x -> x.getState().isOnline())) {
            final ClickhouseTableInfo activeTable = tableStorage.getTable(tableInfo.getType());
            if (activeTable.getTableId().equals(tableInfo.getTableId())) {
                throw new IllegalStateException("this table active");
            }
        }

        removeClickhouseTables(tableInfo);
        return new Response();
    }

    // copied from CleanClickhouseTablesTask
    private void removeClickhouseTables(ClickhouseTableInfo tableInfo) {
        log.info("Cleaning up table: " + tableInfo.getClickhouseFullName());
        String[] parts = tableInfo.getClickhouseFullName().split("\\.");
        if (parts.length != 2) {
            log.warn("Unsupported name of table: " + tableInfo.getClickhouseFullName());
            return;
        }
        String database = parts[0];
        String namePrefix = parts[1];
        if (namePrefix.endsWith(DISTRIB_SUFFIX)) {
            namePrefix = namePrefix.substring(0, namePrefix.length() - DISTRIB_SUFFIX.length());
        }
        // проходимся по каждому хосту
        for (ClickhouseHost host : legacyMdbClickhouseServer.getHosts()) {
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext
                    .useDefaults()
                    .setTimeout(Duration.standardMinutes(10))
                    .setHost(host);
            for (ClickhouseSystemTableInfo table : legacyMdbClickhouseSystemTablesCHDao.getTablesForPrefix(host, database, namePrefix)) {
                // удаляем табличку
                legacyMdbClickhouseServer.execute(ctx,
                        ClickhouseServer.QueryType.INSERT, "DROP TABLE IF EXISTS " + table.getDatabase() + "." + table.getName(),
                        Optional.empty(), Optional.empty());
            }
        }
        legacyClickhouseTablesYDao.delete(tableInfo);
    }

    @Setter(onMethod_ = @RequestQueryProperty)
    @Getter
    public static class Request implements ActionRequest {
        ClickhouseTableInfo newTableInfo;
    }

    public static class Response implements ActionResponse.NormalResponse {
    }
}
