package ru.yandex.webmaster3.worker.delurl;

import NSamovarExportSchema.WebMasterExportFormats;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.addurl.UrlForRecrawl;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.delurl.DelUrlRequest;
import ru.yandex.webmaster3.core.delurl.DelurlState;
import ru.yandex.webmaster3.core.logbroker.reader.IDataProcessing;
import ru.yandex.webmaster3.core.logbroker.reader.MessageContainer;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.*;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.environment.YandexEnvironmentType;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.delurl.DelUrlRequestsService;
import ru.yandex.webmaster3.worker.addurl.RecrawlResult;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

import javax.annotation.PostConstruct;
import java.io.ByteArrayInputStream;
import java.util.*;

/**
 * @author leonidrom
 */

@Slf4j
@Component("delurlResultsProcessingService")
public class DelurlResultsProcessingService implements IDataProcessing {
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(5, Duration.standardSeconds(30));
    private static final int MAX_RETRIES = 2;

    private static final Set<YandexHttpStatus> DEL_URL_CORRECT_STATUSES = Set.of(
            YandexHttpStatus.HTTP_403_FORBIDDEN,
            YandexHttpStatus.HTTP_404_NOT_FOUND,
            YandexHttpStatus.HTTP_410_GONE,
            YandexHttpStatus.HTTP_1003_ROBOTS_TXT_DISALLOW,
            YandexHttpStatus.EXT_HTTP_2005_NOINDEX
    );
    public static final String ROBOT_RESULT_LABEL = "delurl_robot_result";
    public static final String RESULT_LABEL = "delurl_result";

    private final AbtService abtService;
    private final YandexEnvironmentType yandexEnvironmentType;
    private final DelUrlRequestsService delurlRequestsService;
    private final SolomonMetricRegistry solomonMetricRegistry;

    private boolean enableLogging = false;
    private SolomonMetricConfiguration solomonMetricConfiguration;
    private Map<String, SolomonCounter> solomonRobotResultsMetricMap;
    private Map<String, SolomonCounter> solomonResultsMetricMap;

    @Autowired
    public DelurlResultsProcessingService(
            AbtService abtService,
            DelUrlRequestsService delUrlRequestsService,
            @Value("${webmaster3.environmentType}") YandexEnvironmentType yandexEnvironmentType,
            SolomonMetricRegistry solomonMetricRegistry) {
        this.abtService = abtService;
        this.delurlRequestsService = delUrlRequestsService;
        this.yandexEnvironmentType = yandexEnvironmentType;
        this.solomonMetricRegistry = solomonMetricRegistry;
    }

    @PostConstruct
    public void init() {
        enableLogging = yandexEnvironmentType == YandexEnvironmentType.PRODUCTION;
        solomonMetricConfiguration = new SolomonMetricConfiguration();

        solomonRobotResultsMetricMap = new HashMap<>();
        for (var status : WebMasterExportFormats.TFastBanCheckExport.EStatus.values()) {
            String statusStr = status.toString();
            solomonRobotResultsMetricMap.put(statusStr, createSolomonCounter(ROBOT_RESULT_LABEL, statusStr));
        }

        solomonRobotResultsMetricMap.put(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE,
                createSolomonCounter(ROBOT_RESULT_LABEL, SolomonGroupingUtil.AGGREGATE_LABEL_VALUE));

        solomonResultsMetricMap = new HashMap<>();
        for (var status: DelurlState.values()) {
            String statusStr = status.toString();
            solomonResultsMetricMap.put(statusStr, createSolomonCounter(RESULT_LABEL, statusStr));
        }

        solomonResultsMetricMap.put(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE,
                createSolomonCounter(RESULT_LABEL, SolomonGroupingUtil.AGGREGATE_LABEL_VALUE));
    }

    @Override
    public void process(MessageContainer messageContainer) {
        if (enableLogging) {
            log.info("Total messages: {}", messageContainer.getRawMessages().size());
        }

        for (byte[] rawMessage : messageContainer.getRawMessages()) {
            RecrawlResult recrawlResult = parseRecrawlResult(rawMessage);
            if (recrawlResult == null) {
                continue;
            }

            // И тестинг и прод читают из одного топика, разница лишь в том, какие хосты попадают
            // под эксперимент. Чтобы не писать лишнего, делаем здесь фильтрацию.
            WebmasterHostId hostId = recrawlResult.getHostId();
            if (!abtService.isInExperiment(hostId, Experiment.DELURL_SAMOVAR)) {
                continue;
            }

            List<DelUrlRequest> pendingRequests = delurlRequestsService.getPendingRequests(
                    hostId, recrawlResult.getFullUrl());

            if (enableLogging) {
                log.info("Total request pending for {}: {}", recrawlResult.getFullUrl(), pendingRequests.size());
                if (pendingRequests.isEmpty()) {
                    log.error("No pending requests for {}", recrawlResult.getFullUrl());
                }
            }

            List<DelUrlRequest> processedReqs = new ArrayList<>();
            for (var req : pendingRequests) {
                DelurlState newState = computeNewState(req, recrawlResult);
                if (newState == null) {
                    continue;
                }

                if (newState == DelurlState.ERROR && req.getRetriesCount() < MAX_RETRIES) {
                    continue;
                }

                log.info("New state: {}", newState);
                updateResultSolomonCounters(newState);
                processedReqs.add(req.withState(newState));
            }

            if (!processedReqs.isEmpty()) {
                try {
                    RetryUtils.execute(RETRY_POLICY, () -> {
                        delurlRequestsService.updateBatch(processedReqs);
                    });
                } catch (Exception e) {
                    log.error("Failed to update requests states", e);
                }
            }
        }

        messageContainer.commit();
    }

    @Nullable
    private RecrawlResult parseRecrawlResult(byte[] data) {
        try {
            var bais = new ByteArrayInputStream(data);
            var msg = NSamovarExportSchema.WebMasterExportFormats.TFastBanCheckExport.parseFrom(bais);

            if (enableLogging) {
                log.info("Got message:\n{}", msg.toString());
            }

            String fullUrl = msg.getUrl();
            if (enableLogging) {
                log.info("Url: {}", fullUrl);
            }

            Pair<WebmasterHostId, String> pair = null;
            try {
                pair = UrlForRecrawl.toHostIdAndRelativeUrl(fullUrl);
            } catch (Exception e) {
                // игнорируем, обработка ниже
            }

            if (pair == null) {
                if (enableLogging) {
                    log.error("Invalid url: {}", fullUrl);
                }
                return null;
            }

            long timestamp = msg.getCheckSignalTime() * 1000L;
            if (timestamp == 0) {
                timestamp = System.currentTimeMillis();
            }

            WebMasterExportFormats.TFastBanCheckExport.EStatus eStatus = msg.getStatus();
            updateRobotResultSolomonCounters(eStatus);

            if (eStatus != WebMasterExportFormats.TFastBanCheckExport.EStatus.FBS_OK &&
                    eStatus != WebMasterExportFormats.TFastBanCheckExport.EStatus.FBS_URL_NOT_CRAWLED) {
                return null;
            }

            return RecrawlResult.builder()
                    .hostId(pair.getLeft())
                    .relativeUrl(pair.getRight())
                    .fullUrl(fullUrl)
                    .processingTime(new DateTime(timestamp))
                    .code(YandexHttpStatus.parseCode(msg.getHttpCode()))
                    .success(eStatus == WebMasterExportFormats.TFastBanCheckExport.EStatus.FBS_OK)
                    .build();
        } catch (Exception e) {
            if (enableLogging) {
                log.error("Error parsing message \n{}\n", Base64.getEncoder().encodeToString(data), e);
            }

            return null;
        }
    }

    private static DelurlState computeNewState(DelUrlRequest req, RecrawlResult result) {
        if (req.getAddDate().isBefore(result.getProcessingTime())) {
            return checkDelUrlCode(result) ? DelurlState.ACCEPTED : DelurlState.ERROR;
        }

        return null;
    }

    private static boolean checkDelUrlCode(RecrawlResult result) {
        return result.isSuccess() && DEL_URL_CORRECT_STATUSES.contains(result.getCode());
    }

    private SolomonCounter createSolomonCounter(String label, String status) {
        var key = SolomonKey.create(SolomonSensor.LABEL_CATEGORY, "delurl")
                .withLabel(label, status)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.COUNT);

        return solomonMetricRegistry.createCounter(solomonMetricConfiguration, key);
    }

    private void updateRobotResultSolomonCounters(WebMasterExportFormats.TFastBanCheckExport.EStatus status) {
        String statusStr = status.toString();
        solomonRobotResultsMetricMap.get(statusStr).update();
        solomonRobotResultsMetricMap.get(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE).update();
    }

    private void updateResultSolomonCounters(DelurlState state) {
        String statusStr = state.toString();
        solomonResultsMetricMap.get(statusStr).update();
        solomonResultsMetricMap.get(SolomonGroupingUtil.AGGREGATE_LABEL_VALUE).update();
    }
}
