package ru.yandex.webmaster3.worker.delurl;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.delurl.DelUrlRequest;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.delurl.DelUrlRequestsService;
import ru.yandex.webmaster3.storage.delurl.dao.DelUrlQueueYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

/**
 * @author iceflame
 * Отдает проверенные префиксы в антиспам для срочного удаления
 */
@Slf4j
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ExportDelurlPrefixesTask extends PeriodicTask<PeriodicTaskState> {
    private static final String YT_TABLE_PREFIX = "delurl_prefixes.";
    private static final YtSchema YT_TABLE_SCHEMA = new YtSchema();
    private static final YtColumn<String> HOST = YT_TABLE_SCHEMA.addColumn(F.HOST, YtColumn.Type.STRING);
    private static final YtColumn<String> PATH = YT_TABLE_SCHEMA.addColumn(F.PATH, YtColumn.Type.STRING);
    private static final YtColumn<Long> TIMESTAMP = YT_TABLE_SCHEMA.addColumn(F.TIMESTAMP, YtColumn.Type.INT_64);
    private static final YtColumn<Long> USER_ID = YT_TABLE_SCHEMA.addColumn(F.USER_ID, YtColumn.Type.INT_64);

    private final DelUrlQueueYDao delurlQueueYDao;
    private final DelUrlRequestsService delurlRequestsService;
    private final YtService ytService;

    @Value("${webmaster3.worker.delurl.export.rootPath}")
    private YtPath rootPath;

    @Override
    public Result run(UUID runId) throws Exception {
        List<Pair<WebmasterHostId, UUID>> processedRequests = new ArrayList<>();
        YtTableData tableData = ytService.prepareTableData("delurl-prefixes", tableWriter -> {
            try {
                delurlQueueYDao.forEach(pair -> {
                    DelUrlRequest delurlRequest = delurlRequestsService.get(pair.getLeft(), pair.getRight());
                    if (delurlRequest == null) {
                        log.error("Cannot find request {}", pair.getRight());
                        return;
                    }
                    processedRequests.add(pair);
                    HOST.set(tableWriter, IdUtils.hostIdToUrl(delurlRequest.getHostId()));
                    PATH.set(tableWriter, delurlRequest.getRelativeUrl());
                    TIMESTAMP.set(tableWriter, delurlRequest.getAddDate().getMillis());
                    USER_ID.set(tableWriter, delurlRequest.getUserUid());
                    tableWriter.rowEnd();
                });
            } catch (WebmasterYdbException e) {
                throw new WebmasterException("Ydb error", new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
            } catch (YtException e) {
                throw new WebmasterException("Yt error", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            }
        });
        if (processedRequests.isEmpty()) {
            return new Result(TaskResult.SUCCESS);
        }
        // пишем табличку
        ytService.inTransaction(rootPath).execute(cypressService -> {
            // создаем табличку
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(YT_TABLE_SCHEMA);
            YtPath tablePath = YtPath.path(rootPath, YT_TABLE_PREFIX + String.valueOf(System.currentTimeMillis()));
            cypressService.create(tablePath, YtNode.NodeType.TABLE, true, attributes, true);
            // пишем
            cypressService.writeTable(tablePath, tableData);
            return true;
        });
        // удалим из очереди обработанные заявки
        for (Pair<WebmasterHostId, UUID> pair : processedRequests) {
            delurlQueueYDao.delete(pair.getLeft(), pair.getRight());
        }

        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_DELURL_PREFIXES;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 */10 * * * *");
    }

    private interface F {
        String HOST = "Host";
        String PATH = "Path";
        String TIMESTAMP = "Timestamp";
        String USER_ID = "UserId";
    }
}
