package ru.yandex.webmaster3.worker.digest.blog;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Range;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.worker.digest.html.DigestData;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Created by ifilippov5 on 14.11.17.
 */
public class YaBlogsApiService {

    private static final Logger log = LoggerFactory.getLogger(YaBlogsApiService.class);

    private static final long CACHE_EXPIRATION_DAYS = 1;
    private static final int SOCKET_TIMEOUT = 10_000;
    private static final int POSTS_COUNT = 50;
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new WebmasterJsonModule(false))
            .registerModule(new JodaModule())
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES)
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);

    private CloseableHttpClient httpClient;

    private String postsArchiveUrl;
    private TVMTokenService tvmTokenService;

    private LoadingCache<Range<LocalDate>, List<DigestData.BlogPost>> cache = CacheBuilder.newBuilder()
            .expireAfterWrite(CACHE_EXPIRATION_DAYS, TimeUnit.DAYS)
            .build(CacheLoader.from(this::loadBlogPosts));

    public void init() {

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .setConnectionTimeToLive(10, TimeUnit.MINUTES)
                .build();
    }

    private List<DigestData.BlogPost> loadBlogPosts(Range<LocalDate> datesInterval) {
        if (Strings.isNullOrEmpty(postsArchiveUrl)) {
            // not configured
            return Collections.emptyList();
        }

        log.info("Load posts...");
        List<DigestData.BlogPost> blogPosts = new ArrayList<>();
        HttpGet get = new HttpGet(postsArchiveUrl + "?size=" + POSTS_COUNT + "&" + WebmasterBlogPostUtil.tagInQuery());
        get.addHeader(TVMTokenService.TVM2_TICKET_HEADER, tvmTokenService.getToken());

        // загружаем посты всего раз, поэтому ретраим
        try {
            RetryUtils.query(RETRY_POLICY, () -> {
                try (CloseableHttpResponse response = httpClient.execute(get)) {
                        if (response.getStatusLine().getStatusCode() / 100 == 2) {
                            log.info("Request success");

                            Iterator<JsonNode> root = OM.readTree(response.getEntity().getContent()).elements();
                            while (root.hasNext()) {
                                JsonNode node = root.next();
                                WebmasterBlogPostUtil.parseBlogPost(node, datesInterval, blogPosts::add);
                            }
                            return true;
                        } else {
                            String responseString = IOUtils.toString(response.getEntity().getContent(), Charsets.UTF_8);
                            log.error("Error from remote service (" + response.getStatusLine().getStatusCode() + "): " + responseString);
                            throw new RuntimeException("Error from blogs api service");
                        }
                } catch (IOException e) {
                    throw new WebmasterException("Unable to query " + get,
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
                }
            });
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        return blogPosts;
    }

    public List<DigestData.BlogPost> getBlogPosts(Range<LocalDate> datesInterval) {
        return cache.getUnchecked(datesInterval);
    }

    @Required
    public void setPostsArchiveUrl(String postsArchiveUrl) {
        this.postsArchiveUrl = postsArchiveUrl;
    }

    @Required
    public void setTvmTokenService(TVMTokenService tvmTokenService) {
        this.tvmTokenService = tvmTokenService;
    }
}
