package ru.yandex.webmaster3.worker.digest.checklist;

import java.util.List;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.digest.DigestSchedulerUtil;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.checklist.service.AllHostsProblemsService;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 08.09.17.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UploadSiteProblemsForDigestTask extends PeriodicTask<PeriodicTaskState> {
    public static final DateTimeFormatter DATE_FORMAT_ONLY_DAY = DateTimeFormat.forPattern("yyyyMMdd");
    private static final String YT_TABLE_PREFIX = "";
    public static final Pattern RESULT_TABLE_NAME_PATTERN = Pattern.compile(YT_TABLE_PREFIX + "(\\d+)");

    private final AllHostsProblemsService allHostsProblemsService;
    private final YtService ytService;
    @Value("${external.yt.service.arnold.root.default}/checklist/site-problems")
    private YtPath workDir;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.DIGEST_SITE_PROBLEMS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("10 0 * * * *");
    }

    @Override
    public PeriodicTask.Result run(UUID runId) throws Exception {
        DateTime lastDigestDate = DigestSchedulerUtil.getLastDigestBefore(DateTime.now());
        log.debug("Last digest date : {}", lastDigestDate.toString(DATE_FORMAT_ONLY_DAY));

        if (lastDigestDataAlreadyExists(lastDigestDate.toLocalDate())) {
            log.debug("Table with last digest data already exists");
            return new PeriodicTask.Result(TaskResult.SUCCESS);
        }

        String ytTableSuffix = lastDigestDate.toString(DATE_FORMAT_ONLY_DAY);
        String tableName = YT_TABLE_PREFIX + ytTableSuffix;
        YtPath finishPath = YtPath.path(workDir, tableName);

        ytService.inTransaction(finishPath).execute(cypressService -> {
            cypressService.create(finishPath, YtNode.NodeType.TABLE, true, new YtNodeAttributes().setSchema(F.SCHEMA), true);
            cypressService.writeTable(finishPath, tableWriter -> {
                allHostsProblemsService.forEachProblem(problemInfo -> {
                    if (problemInfo.getState().isPresent()) {
                        F.HOST.set(tableWriter, problemInfo.getHostId().toString());
                        F.PROBLEM_TYPE.set(tableWriter, (long) problemInfo.getProblemType().value());
                        tableWriter.rowEnd();
                    }
                });
            });
            cypressService.waitFor(cypressService.sort(finishPath, finishPath, F.HOST.getName(), F.PROBLEM_TYPE.getName()));
            return true;
        });

        return new PeriodicTask.Result(TaskResult.SUCCESS);
    }

    private boolean lastDigestDataAlreadyExists(LocalDate lastDigestDate) throws InterruptedException {
        return ytService.withoutTransactionQuery(cypressService -> {
            if (!cypressService.exists(workDir)) {
                return false;
            }
            List<YtPath> resultTables = findResultTables(cypressService);
            if (resultTables.isEmpty()) {
                return false;
            }
            // проверим дату последней таблицы
            YtPath lastTable = resultTables.get(resultTables.size() - 1);
            Matcher matcher = RESULT_TABLE_NAME_PATTERN.matcher(lastTable.getName());
            if (!matcher.matches()) {
                log.error("Invalid table name : {}", lastTable.getName());
                return false;
            }
            LocalDate tableDate = LocalDate.parse(matcher.group(1), DATE_FORMAT_ONLY_DAY);
            log.debug("Latest table suffix : {}", tableDate.toString(DATE_FORMAT_ONLY_DAY));
            return !tableDate.isBefore(lastDigestDate);
        });
    }

    private List<YtPath> findResultTables(YtCypressService cypressService) throws YtException {
        return cypressService.list(workDir).stream()
                .filter(table -> RESULT_TABLE_NAME_PATTERN.matcher(table.getName()).matches())
                .sorted().collect(Collectors.toList());
    }

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> HOST = SCHEMA.addColumn("Host", YtColumn.Type.STRING);
        YtColumn<Long> PROBLEM_TYPE = SCHEMA.addColumn("Problem", YtColumn.Type.INT_64);
    }
}
