package ru.yandex.webmaster3.worker.digest.html;

import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.digest.graphics.draw.ChartSettings;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.core.util.json.polymorphic.Discriminator;
import ru.yandex.webmaster3.storage.achievements.model.AchievementService;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;
import ru.yandex.webmaster3.storage.achievements.model.official.AchievementOfficialType;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.tanker.I18nDigestAchievement;
import ru.yandex.webmaster3.tanker.I18nDigestChecklist;
import ru.yandex.webmaster3.tanker.I18nDigestClicks;
import ru.yandex.webmaster3.tanker.I18nDigestDomains;
import ru.yandex.webmaster3.tanker.I18nDigestDynamic;
import ru.yandex.webmaster3.tanker.I18nDigestFixedSiteProblems;
import ru.yandex.webmaster3.tanker.I18nDigestHostEvents;
import ru.yandex.webmaster3.tanker.I18nDigestMetrikaCounterProblem;
import ru.yandex.webmaster3.tanker.I18nDigestMirrorTexts;
import ru.yandex.webmaster3.tanker.I18nDigestPresentSiteProblems;
import ru.yandex.webmaster3.tanker.I18nDigestReviewTexts;
import ru.yandex.webmaster3.tanker.I18nDigestSearchable;
import ru.yandex.webmaster3.tanker.I18nDigestTemp;
import ru.yandex.webmaster3.tanker.I18nEmailTexts;
import ru.yandex.webmaster3.tanker.I18nLiteDigestTexts;
import ru.yandex.webmaster3.tanker.WebmasterLinksProvider;
import ru.yandex.webmaster3.tanker.digest.html.CSS;
import ru.yandex.webmaster3.tanker.digest.html.HtmlNode;
import ru.yandex.webmaster3.tanker.digest.html.HtmlSize;
import ru.yandex.webmaster3.worker.digest.DigestAttachType;

import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.ACHIEVEMENTS;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.CHECKLIST;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.CLICKS_CHART;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.IMPORTANT_PAGES;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.QUERIES;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.QUERIES_GROUPS;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.RECOMMENDED_URLS;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.REVIEWS;
import static ru.yandex.webmaster3.worker.digest.html.DigestBlockType.SEARCHABLE;

/**
 * Created by ifilippov5 on 22.09.17.
 */
public class DigestBlocksBuilder {
    private static final Logger log = LoggerFactory.getLogger(DigestBlocksBuilder.class);

    public static final int MAIL_WIDTH = 480;
    public static final String CUSTOM_BLOCK_PLACEHOLDER = "${CUSTOM_BLOCK_PLACEHOLDER}";
    public static final String CUSTOM_BLOCK_PLACEHOLDER_HOST = "${CUSTOM_BLOCK_PLACEHOLDER_HOST_ID}";

    private static HtmlNode.TextNode NBSP = HtmlNode.safeText("&nbsp;");
    private static HtmlNode.TextNode BULL = HtmlNode.safeText("&bull;");
    private static HtmlNode.TextNode BLACK_CIRCLE = HtmlNode.text("\u25CF ");

    private final NotificationType notificationType;
    private final DigestData.Host host;
    private final LocalDate digestFrom;
    private final LocalDate digestTo;
    private final DigestI18n i18n;
    private final WebmasterLinksProvider webmasterLinksProvider;
    private final Set<DigestAttachType> attaches;
    private final ChartSettings chartSettings;
    private final AchievementService achievementService;
    private final boolean forced;

    private static final String UTM_TERM_DASHBOARD_FOOTER = "dashboard_footer";
    private static final String UTM_TERM_UNSUBSCRIBE = "unsubscribe";
    private static final String UTM_TERM_MESSAGE_TITLE = "message_title";
    private static final String UTM_TERM_MIRROR = "mirror";
    private static final String UTM_TERM_LITE_DIGEST_DESCRIPTION = "lite_digest_description";

    private static final String UTM_TERM_DIGEST_CUSTOM_BLOCK = "digest-custom-block";

    public DigestBlocksBuilder(DigestData.Host host, LocalDate digestFrom, LocalDate digestTo,
                               DigestI18n i18n, WebmasterLinksProvider webmasterLinksProvider,
                               Set<DigestAttachType> attaches, ChartSettings chartSettings,
                               NotificationType notificationType, AchievementService achievementService, boolean forced
    ) {
        this.host = host;
        this.digestFrom = digestFrom;
        this.digestTo = digestTo;
        this.i18n = i18n;
        this.webmasterLinksProvider = webmasterLinksProvider;
        this.attaches = attaches;
        this.chartSettings = chartSettings;
        this.notificationType = notificationType;
        this.achievementService = achievementService;
        this.forced = forced;
    }

    private static HtmlNode bNumber(long value) {
        return HtmlNode.text(DigestTextUtil.formatNumber(value));
    }

    public HtmlNode.TableTag bTable() {
        return new HtmlNode.TableTag()
                .border(HtmlSize.px(0))
                .cellpadding(HtmlSize.px(0))
                .cellspacing(HtmlSize.px(0))
                .addStyle(Styles.TABLE);
    }

    public DigestBlocks build(DigestData data) {
        List<HtmlNode> sections = new ArrayList<>();
        List<DigestBlockType> blockTypes = new ArrayList<>();

        addBlock(sections, blockTypes, ACHIEVEMENTS,() -> new ChecklistBlocks().bAchievements(data.getAchievements()));
        addBlock(sections, blockTypes, CHECKLIST,   () -> new ChecklistBlocks().bChecklist(data.getChecklist()));
        addBlock(sections, blockTypes, SEARCHABLE,() -> new SearchableBlocks().bSearchable(data.getSearchable()));
        addBlock(sections, blockTypes, IMPORTANT_PAGES,() -> new ImportantPagesBlocks().bImportant(data.getImportantUrl()));
        addBlock(sections, blockTypes, RECOMMENDED_URLS,() -> new RecommendedUrlsBlocks().bRecommended(data.getRecommendedUrl()));
        addBlock(sections, blockTypes, CLICKS_CHART,() -> new ClicksChartBlocks().bClicksChart(data.getClicks(), data.getPrevWeekFrom(), data.getPrevWeekTo()));
        addBlock(sections, blockTypes, QUERIES,() -> new QueriesBlocks().bQueries(data.getQueries()));
        addBlock(sections, blockTypes, QUERIES_GROUPS,() -> new QueriesGroupsBlocks().bQueriesGroups(data.getQueriesGroups()));
        addBlock(sections, blockTypes, REVIEWS,() -> new ReviewsBlock().bReviews(data.getReviews()));

        // Temporary remove metrika problem block
        //addBlock(sections, blockTypes, METRIKA_COUNTER_PROBLEM,() -> new MetrikaCounterProblemBlock().bMetrikaCounterProblem(data.getMetrikaCounterProblem()));

        if (sections.size() < 2) { // Минимум 2 блока
            return DigestBlocks.empty();
        }

        return new DigestBlocks(
                bTable()
                        .addStyle(Styles.CONTAINER)
                        .addRow(
                                HtmlNode.td(
                                        bLogo(),
                                        bMainHeader(),
                                        bIsMirrorBlock(data.getMirrors())
                                        )
                                        .addContent(
                                                sectionWithDelimiter(new TempNotificationInfo().bNotificationInfo()) //TODO выпилить после дайджеста
                                        )
                                        .addContent(sections)
                                        .addContent(sectionWithDelimiter(isFullDigest() ? bFullDigestCustomBlock() : HtmlNode.EMPTY))
                                        .addContent(
                                                sectionWithDelimiter(
                                                        isFullDigest() ? new BlogPostsBlocks().bBlogPosts(data.getBlogPosts()) : HtmlNode.EMPTY
                                                )
                                        )
                                        .addContent(bFooter())
                                        .addStyle(Styles.MAIN_TABLE)
                        )
                ,
                blockTypes
        );
    }

    private void addBlock(List<HtmlNode> sections, List<DigestBlockType> blockTypes, DigestBlockType type, Supplier<HtmlNode> supplier){
        HtmlNode block = sectionWithDelimiter(supplier.get());

        //Если блок не пуст добавляем
        if(!contentIsEmpty(block)){
            sections.add(block);
            blockTypes.add(type);
        }
    }

    private static HtmlNode sectionWithDelimiter(HtmlNode section) {
        if (section == HtmlNode.EMPTY) {
            return HtmlNode.EMPTY;
        } else {
            return HtmlNode.list(section, bDelimiter());
        }
    }

    public HtmlNode bMainHeader() {
        return HtmlNode.div()
                .addStyle(Styles.MAIN_HEADER)
                .addContent(
                        HtmlNode.h1(
                                i18n.getDigestTitle(
                                        HtmlNode.a(webmasterLinksProvider.hostDashboard(host.getId(), UTM_TERM_MESSAGE_TITLE))
                                                .addContent(HtmlNode.text(host.getDisplayName()))
                                                .addStyle(Styles.MAIN_HOST_LINK)
                                )
                        ).addStyle(Styles.MAIN_TITLE),
                        HtmlNode.p()
                                .addStyle(Styles.MAIN_PERIOD)
                                .addContent(
                                        HtmlNode.text(
                                                I18nDigestDynamic.DIGEST_PERIOD.newBuilder(i18n.language)
                                                        .dateRange(DigestTextUtil.dateRangeInHeader(i18n.language, digestFrom, digestTo))
                                                        .render()
                                        )
                                )
                );
    }


    public HtmlNode bIsMirrorBlock(Optional<DigestData.Mirrors> mainHostOpt) {
        if (mainHostOpt.isEmpty() || StringUtils.isEmpty(mainHostOpt.get().getMainHost())) {
            return HtmlNode.EMPTY;
        }
        String mainHost = mainHostOpt.get().getMainHost();
        WebmasterHostId mainHostId = IdUtils.urlToHostId(mainHost);
        return HtmlNode.div()
                .addStyle(Styles.MAIN_HEADER)
                .addContent(
                        HtmlNode.p(
                                I18nDigestMirrorTexts.HTML_mirror_title.newBuilder(i18n.language)
                                .mirror_host(
                                        HtmlNode.a(webmasterLinksProvider.hostDashboard(host.getId(), UTM_TERM_MIRROR))
                                                .addStyle(Styles.LANDING_LINK)
                                                .addContent(HtmlNode.text(host.getDisplayName()))
                                )
                                .main_host(
                                        HtmlNode.a(webmasterLinksProvider.hostDashboard(mainHostId, UTM_TERM_MIRROR))
                                                .addStyle(Styles.LANDING_LINK)
                                                .addContent(HtmlNode.text(mainHost))
                                ).add_host_link(
                                        HtmlNode.a(webmasterLinksProvider.hostDashboard(mainHostId, UTM_TERM_MIRROR))
                                                .addStyle(Styles.LANDING_LINK)
                                                .addContent(HtmlNode.text(I18nDigestMirrorTexts.mirror_add_host.getText(i18n.language)))
                                )
                        )
                );
    }

    public HtmlNode bLiteDigestDescriptionBlock() {
        return HtmlNode.div()
                .addStyle(Styles.MAIN_HEADER)
                .addContent(
                        HtmlNode.p(
                                I18nLiteDigestTexts.HTML_lite_digest_description.newBuilder(i18n.language)
                                        .host(
                                                HtmlNode.a(webmasterLinksProvider.hostDashboard(host.getId(), UTM_TERM_LITE_DIGEST_DESCRIPTION))
                                                        .addStyle(Styles.LANDING_LINK)
                                                        .addContent(HtmlNode.text(host.getDisplayName()))
                                        )
                                        .subscribe(
                                                HtmlNode.a(webmasterLinksProvider.notificationSettings(UTM_TERM_LITE_DIGEST_DESCRIPTION))
                                                        .addStyle(Styles.LANDING_LINK)
                                                        .addContent(HtmlNode.text(I18nLiteDigestTexts.lite_digest_subscribe.getText(i18n.language)))
                                        )
                        )
                );
    }

    public HtmlNode bFullDigestCustomBlock() {
        return new HtmlNode.TextNode(CUSTOM_BLOCK_PLACEHOLDER, false);
    }

    public HtmlNode bLogo() {
        HtmlNode.ImgTag img = new HtmlNode.ImgTag(i18n.getLogoLink())
                .alt(i18n.getDigestLogoText())
                .width(HtmlSize.px(160));
        return bTable()
                .addRow(
                        HtmlNode.td(img)
                                .addStyle(Styles.LOGO)
                );
    }

    private static HtmlNode bDelimiter() {
        return new HtmlNode.HrTag()
                .addStyle(Styles.DELIMITER);
    }

    public HtmlNode bSection(HtmlNode... content) {
        if (contentIsEmpty(content)) {
            return HtmlNode.EMPTY;
        }
        return HtmlNode.div()
                .addStyle(Styles.SECTION)
                .addContent(content);
    }

    private boolean contentIsEmpty(HtmlNode... content) {
        for (HtmlNode node : content) {
            if (node != HtmlNode.EMPTY) {
                return false;
            }
        }
        return true;
    }

    public class TempNotificationInfo {
        public static final String UTM_TERM_UNSUBSCRIBE_FORCED = "unsubscribe_from_forced";

        public HtmlNode bNotificationInfo() {
            if (!forced || i18n.language != LanguageEnum.RU) {
                return HtmlNode.EMPTY;
            }

            return bSection(
                    HtmlNode.p(HtmlNode.text(I18nDigestTemp.FORCED_NOTIFICATION_INFO.getText(i18n.language)))
            );
        }
    }

    public class ChecklistBlocks {
        public final String UTM_TERM_CHECKLIST_ACHIEVEMENTS = "checklist_achievements";
        public final String UTM_TERM_CHECKLIST_PROBLEMS = "checklist_problems";
        public final String UTM_TERM_CHECKLIST_CHANGES = "checklist_changes";
        public final String UTM_TERM_TURBO_PROBLEMS = "turbo_problems";
        public final String UTM_TERM_TURBO_AUTOPARSER = "autoturbo";
        public final String UTM_TERM_IKS = "sqi";
        public final String UTM_TERM_TURBO_YML = "yml";

        public HtmlNode bChecklist(Optional<DigestData.Checklist> checklistOpt) {
            return checklistOpt.map(checklist -> {
                        HtmlNode presentProblems = bPresentProblems(checklist.getPresentProblems(), checklist.isProblemsOnlyInTurbo(), checklist.isProblemsOnlyInTurboContent());
                        HtmlNode bFixedProblems = bFixedProblems(checklist.getFixedProblems());
                        HtmlNode bEvents = bEvents(checklist.getEvents());

                        if (presentProblems == HtmlNode.EMPTY && bFixedProblems == HtmlNode.EMPTY && bEvents == HtmlNode.EMPTY){
                            return HtmlNode.EMPTY;
                        }

                        return bSection(
                                bSectionTitle(HtmlNode.text(I18nDigestChecklist.changes.getText(i18n.language))),
                                presentProblems,
                                bFixedProblems,
                                bEvents
                        );
                    }
            ).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode bAchievements(Optional<List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>>> achievements) {
            return achievements.map(a -> bSection(bAchievements(a))).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode bAchievements(List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>> achievements) {
            Map<AchievementType, Pair<List<AchievementTld>, DigestData.AbstractAchievement>> achievementsByType = new HashMap<>();
            Map<AchievementOfficialType, Pair<List<AchievementTld>, DigestData.AbstractAchievement>> officialAchievementsByType = new HashMap<>();

            for (Pair<AchievementTld, List<DigestData.AbstractAchievement>> achievement : achievements) {
                for (DigestData.AbstractAchievement abstractAchievement : achievement.getSecond()) {
                    if (abstractAchievement instanceof DigestData.AchievementOfficial){
                        DigestData.AchievementOfficial ach = (DigestData.AchievementOfficial) abstractAchievement;
                        if (officialAchievementsByType.containsKey(ach.type)){
                            officialAchievementsByType.get(ach.type).first.add(achievement.first);
                        }else{
                            List<AchievementTld> achs = new ArrayList<>();
                            achs.add(achievement.first);
                            officialAchievementsByType.put(ach.type, Pair.of(achs,abstractAchievement));
                        }
                    }
                    if (abstractAchievement instanceof DigestData.Achievement){
                        DigestData.Achievement ach = (DigestData.Achievement) abstractAchievement;
                        if (achievementsByType.containsKey(ach.type)){
                            achievementsByType.get(ach.type).first.add(achievement.first);
                        }else{
                            List<AchievementTld> achs = new ArrayList<>();
                            achs.add(achievement.first);
                            achievementsByType.put(ach.type, Pair.of(achs,abstractAchievement));
                        }
                    }
                }
            }
            List<HtmlNode> nodes = new ArrayList<>();
            achievementsByType.forEach((k,v)->nodes.add(bAchievement(k,v)));
            officialAchievementsByType.forEach((k,v)->nodes.add(bAchievement(k,v)));

            List<HtmlNode> filteredNodes = nodes.stream().filter(a -> a!=HtmlNode.EMPTY).collect(Collectors.toList());

            if (filteredNodes.isEmpty()) {
                return HtmlNode.EMPTY;
            }

            return bSection(
                    bSectionTitle(
                            HtmlNode.text(I18nDigestAchievement.title.getText(i18n.language))
                    ),
                    bChecklistChunk(
                            filteredNodes,
                            Colors.GREEN.getValue(),
                            "",
                            ""
                    )
            );
        }

        private HtmlNode bAchievement(Discriminator achType, Pair<List<AchievementTld>, DigestData.AbstractAchievement> achievementTld) {
            DigestData.AbstractAchievement achievement = achievementTld.second;
            List<AchievementTld> tlds = achievementTld.first;

            List<String> domains = new ArrayList<>();

            for (AchievementTld tld : tlds) {
                switch (tld) {
                    case RU:
                        webmasterLinksProvider.achievementLink(host.getId(), UTM_TERM_CHECKLIST_ACHIEVEMENTS);
                        domains.add(I18nDigestDomains.ru.getText(i18n.language));
                        break;
                    case BY:
                        domains.add(I18nDigestDomains.by.getText(i18n.language));
                        break;
                    case KZ:
                        domains.add(I18nDigestDomains.kz.getText(i18n.language));
                        break;
                    case UZ:
                        domains.add(I18nDigestDomains.uz.getText(i18n.language));
                        break;
                    case UA:
                        domains.add(I18nDigestDomains.ua.getText(i18n.language));
                        break;
                    default:
                        log.warn("Unsupported domain: {}", achievement.domain);
                        return HtmlNode.EMPTY;
                }
            }

            String title = null;
            if (achType instanceof AchievementOfficialType) {
                AchievementOfficialType type = (AchievementOfficialType) achType;
                if (!achievementService.isDisabled(type)) {
                    if (achievement instanceof DigestData.BrandedOfficialAchievement) {
                        String brand = ((DigestData.BrandedOfficialAchievement)achievement).brand;
                        switch (type) {
                            case AUTO:
                                title = I18nDigestAchievement.auto.newBuilder(i18n.language).brand(brand).render();
                                break;
                            case BRAND:
                                // TODO not yet specified
                                break;
                            case SERVICE_CENTER:
                                // TODO not yet specified
                                break;
                        }
                    } else {
                        switch (type) {
                            case MFO:
                                title = I18nDigestAchievement.mfo.getText(i18n.language);
                                break;
                            case SSD:
                                title = I18nDigestAchievement.ssd.getText(i18n.language);
                                break;
                            case GOV:
                                title = I18nDigestAchievement.official.getText(i18n.language);
                                break;
                            case YANDEX:
                                title = I18nDigestAchievement.yandex.getText(i18n.language);
                                break;
                            case CREDIT_ORGANIZATION:
                                title = I18nDigestAchievement.cbr.getText(i18n.language);
                                break;
                            case AIRLINE:
                                title = I18nDigestAchievement.avia.getText(i18n.language);
                                break;
                            case EMBASSY:
                                title = I18nDigestAchievement.embassy.getText(i18n.language);
                                break;
                            case VISA_CENTER:
                                title = I18nDigestAchievement.visa_center.getText(i18n.language);
                                break;
                        }
                    }
                }
            } else if (achievement instanceof DigestData.Achievement) {
                AchievementType type = ((DigestData.Achievement)achievement).type;
                if (!achievementService.isDisabled(type)) {
                    switch (type) {
                        case USER_CHOICE:
                            title = I18nDigestAchievement.user_choice.getText(i18n.language);
                            break;
                        case POPULAR:
                            title = I18nDigestAchievement.popular.getText(i18n.language);
                            break;
                        case HTTPS:
                            title = I18nDigestAchievement.https.getText(i18n.language);
                            break;
                        case TAS_IX:
                            title = I18nDigestAchievement.tasix.getText(i18n.language);
                            break;
                        case TURBO:
                            title = I18nDigestAchievement.turbo.getText(i18n.language);
                            break;
                    }
                }
            }

            if (domains.size() > 0 && !StringUtils.isEmpty(title)) {
                HtmlNode achNode = HtmlNode.a(webmasterLinksProvider.achievementLink(host.getId(), UTM_TERM_CHECKLIST_ACHIEVEMENTS))
                        .addStyle(Styles.LANDING_LINK).addContent(HtmlNode.safeText(title));
                HtmlNode domainsNode = new HtmlNode.TextNode(StringUtils.join(domains, ", "), true);

                switch (domains.size()){
                    case 1:
                        return I18nDigestAchievement.HTML_aquired.newBuilder(i18n.language)
                                .achievement(achNode)
                                .domain(domainsNode);
                    case 5:
                        return I18nDigestAchievement.HTML_aquired_without_region.newBuilder(i18n.language)
                                .achievement(achNode);
                    default:
                        return I18nDigestAchievement.HTML_aquired_many.newBuilder(i18n.language)
                                .achievement(achNode)
                                .domain(domainsNode);
                }
            }

            return HtmlNode.EMPTY;
        }

        private HtmlNode bPresentProblems(List<SiteProblemTypeEnum> presentProblems, boolean problemsOnlyInTurbo, boolean problemsOnlyInTurboContent) {
            return bChecklistChunk(
                        presentProblems.stream()
                                .map(problem -> HtmlNode.safeText(I18nDigestPresentSiteProblems.fromEnum(problem).getText(i18n.language)))
                                .collect(Collectors.toList()),
                        Colors.RED.getValue(),
                        i18n.getChecklistProblems(),
                        webmasterLinksProvider.checklistLink(host.getId(), UTM_TERM_CHECKLIST_PROBLEMS, problemsOnlyInTurbo, problemsOnlyInTurboContent, UTM_TERM_TURBO_PROBLEMS)
            );
        }

        private HtmlNode bFixedProblems(List<SiteProblemTypeEnum> fixedProblems) {
            return bChecklistChunk(
                        fixedProblems.stream()
                                .map(problem -> HtmlNode.safeText(I18nDigestFixedSiteProblems.fromEnum(problem).getText(i18n.language)))
                                .collect(Collectors.toList()),
                        Colors.GREEN.getValue(),
                        i18n.getChecklistChecks(),
                        webmasterLinksProvider.checklistOverallLink(host.getId(), UTM_TERM_CHECKLIST_CHANGES)
            );
        }

        private HtmlNode bEvents(List<DigestData.HostEvent> events) {
            return bChecklistChunk(
                    events.stream().map(this::bHostEvent).filter(a -> a != HtmlNode.EMPTY).collect(Collectors.toList()),
                    Colors.GRAY.getValue(),
                    "",
                    ""
            );
        }

        private HtmlNode bHostEvent(DigestData.HostEvent event) {
            if (event instanceof DigestData.RecrawledPages) {
                return I18nDigestHostEvents.HTML_PAGES_RECRAWLED.newBuilder(i18n.language)
                        .count(((DigestData.RecrawledPages) event).count, DigestBlocksBuilder::bNumber);
            } else if (event instanceof DigestData.HostDisplayNameChanged) {
                return HtmlNode.safeText(I18nDigestHostEvents.DISPLAY_NAME_CHANGED.getText(i18n.language));
            } else if (event instanceof DigestData.RegionChanged) {
                return HtmlNode.safeText(I18nDigestHostEvents.REGION_CHANGED.getText(i18n.language));
            } else if (event instanceof DigestData.UserVerifiedHost) {
                return HtmlNode.text(I18nDigestHostEvents.USER_VERIFIED_HOST.newBuilder(i18n.language)
                        .login(((DigestData.UserVerifiedHost) event).login).render());
            } else if (event instanceof DigestData.IksUpdated) {
                String owner = ((DigestData.IksUpdated) event).owner;
                return I18nDigestHostEvents.HTML_IKS_ON_OWNER_CHANGED.newBuilder(i18n.language)
                        .supportIksLink(HtmlNode.a(webmasterLinksProvider.supportLinkIKS())
                                .addStyle(Styles.LANDING_LINK)
                                .addContent(HtmlNode.text(I18nEmailTexts.IKS.newBuilder(i18n.language).utm(webmasterLinksProvider.getUTMLabels()).render())))
                        .owner(HtmlNode.a(webmasterLinksProvider.addSchema(IdUtils.IDN.toUnicode(owner)))
                                .addStyle(Styles.LANDING_LINK)
                                .addContent(HtmlNode.safeText(IdUtils.IDN.toUnicode(owner))))
                        .iksValue(HtmlNode.safeText(String.valueOf(((DigestData.IksUpdated) event).iksValue)))
                        .iksLink(HtmlNode.a(webmasterLinksProvider.iksLink(host.getId(), UTM_TERM_IKS))
                                .addStyle(Styles.LANDING_LINK)
                                .addContent(HtmlNode.text(I18nEmailTexts.IKS_QUALITY.newBuilder(i18n.language).utm(webmasterLinksProvider.getUTMLabels()).render())))
                        .improveIksLink(HtmlNode.a(webmasterLinksProvider.improveIKSLink())
                                .addStyle(Styles.LANDING_LINK)
                                .addContent(HtmlNode.text(I18nEmailTexts.IKS_RECOMMENDATIONS.newBuilder(i18n.language).utm(webmasterLinksProvider.getUTMLabels()).render())))
                        .brTag(HtmlNode.br());
            }
            return HtmlNode.EMPTY;
        }

        public HtmlNode bChecklistChunk(List<HtmlNode> checklistChunkData, String dotColor, String linkText, String linkUrl) {
            if (checklistChunkData.isEmpty()) {
                return HtmlNode.EMPTY;
            }
            return HtmlNode.div()
                    .addStyle(Styles.CHECKLIST_CHUNK)
                    .addContent(
                            bChecklistList(checklistChunkData, dotColor),
                            bLandingLink(HtmlNode.safeText(linkText), linkUrl)
                    );
        }

        public HtmlNode bChecklistList(List<HtmlNode> items, String dotColor) {
            return HtmlNode.ul(
                    items.stream().map(item -> bChecklistItem(item, dotColor)).collect(Collectors.toList())
            ).addStyle(Styles.CHECKLIST_LIST);
        }

        public HtmlNode.LiTag bChecklistItem(HtmlNode text, String dotColor) {
            return new HtmlNode.LiTag()
                            .addStyle(Styles.CHECKLIST_ITEM)
                            .addContent(
                                    bDot(dotColor),
                                    text
                            );
        }
    }

    public class SearchableBlocks {
        public final String UTM_TERM_ALL_SEARCH_PAGES = "all_search_pages";

        public HtmlNode bSearchable(Optional<DigestData.Searchable> searchableOpt) {
            if (!searchableOpt.isPresent()) {
                return HtmlNode.EMPTY;
            }
            DigestData.Searchable searchable = searchableOpt.get();
            return bSection(
                    bSectionTitle(
                            I18nDigestSearchable.HTML_title
                                    .newBuilder(i18n.language)
                                    .count(searchable.getCount(), DigestBlocksBuilder::bNumber)
                    )
                    .addContent(HtmlNode.text("  "))
                    .addContent(bGrowth(searchable.getDifference(), true)),
                    bSearchableSamples(searchable.getSamples()),
                    bLandingLink(
                            HtmlNode.safeText(i18n.getSearchableLink()),
                            webmasterLinksProvider.searchPages(host.getId(), UTM_TERM_ALL_SEARCH_PAGES)
                    )
            );
        }

        public HtmlNode bSearchableSamples(List<DigestData.Searchable.Sample> samples) {
            if (samples.isEmpty()) {
                return HtmlNode.EMPTY;
            }
            boolean showReason = samples.stream().anyMatch(sample -> sample.getStatus() == DigestData.StatsEnum.REMOVED);
            List<HtmlNode.ThTag> headers = new ArrayList<>();
            headers.add(thLeft(HtmlNode.safeText(i18n.getSearchableTableUrl()))
                    .styleWidth(HtmlSize.px((int)(MAIL_WIDTH * 0.6))));
            if (showReason) {
                headers.add(thRight(HtmlNode.safeText(i18n.getSearchableTableReason()))
                        .styleWidth(HtmlSize.px((int)(MAIL_WIDTH * 0.2))));
            }
            headers.add(thRight(HtmlNode.safeText(i18n.getSearchableTableStatus()))
                    .styleWidth(HtmlSize.px((int)(MAIL_WIDTH * 0.2))));

            return bTable()
                    .addHeader(headers)
                    .addRows(
                            samples.stream().map(sample ->  bSearchablePage(sample, showReason)).collect(Collectors.toList())
                    );
        }

        public HtmlNode.TrDTag bSearchablePage(DigestData.Searchable.Sample link, boolean showReason) {
            List<HtmlNode.TdTag> cells = new ArrayList<>();
            cells.add(tdLeft(
                    link.getTitle().isEmpty() ? HtmlNode.EMPTY :
                            HtmlNode.p(HtmlNode.text(link.getTitle()))
                                    .addStyle(Styles.SEARCHABLE_TITLE)
                                    .addStyle(Styles.WRAP_BREAK_WORDS),
                    HtmlNode.p(HtmlNode.text(URLDecoder.decode(link.getUrl(), Charset.defaultCharset())))
                            .addStyle(Styles.SEARCHABLE_URL)
                            .addStyle(Styles.WRAP_ELLIPSIS)
                            .styleWidth(HtmlSize.px((int)(MAIL_WIDTH * 0.6)))
            ));
            if (showReason){
                cells.add(tdRight(HtmlNode.text(link.getReason()))
                        .addStyle(Styles.SEARCHABLE_REASON));
            }
            cells.add(tdRight(HtmlNode.text(i18n.getSearchable(link.getStatus())))
                    .addStyle(Styles.getSearchable(link.getStatus())));
            return HtmlNode.trd(cells);
        }

        // TODO нужно порефакторить этот метод, после того как этот блок будет добавлен в письмо и появятся тесты
        public HtmlNode bSearchableBar(DigestData.Searchable.Stats stats) {
            long added = stats.getAdded().getValue();
            long removed = stats.getRemoved().getValue();
            long total = added + removed;
            if (total == 0) {
                return HtmlNode.EMPTY;
            }

            List<HtmlNode.TdTag> titles = new ArrayList<>();
            List<HtmlNode.TdTag> strips = new ArrayList<>();
            int addedPercent = Math.max(2, (int) ((((float) added) / total) * 100));
            int removedPercent = Math.max(2, (int) ((((float) removed) / total) * 100));
            log.info("Added: {}, removed: {}", addedPercent, removedPercent);

            if (added > 0) {
                titles.add(
                        tdLeft(
                                bDot(Colors.SEARCHABLE_ADDED.getValue()),
                                HtmlNode.safeText(i18n.getSearchableAdded(DigestTextUtil.formatNumber(stats.getAdded().getView())))
                        ).addStyle(Styles.B_SEARCHABLE_BAR_STATS)
                );

                strips.add(
                        HtmlNode.td(NBSP)
                                .addStyle(Styles.B_SEARCHABLE_BAR_STRIP)
                                .addStyle("background-color", Colors.SEARCHABLE_ADDED.getValue())
                                .addStyle("width", String.valueOf(addedPercent) + "%")
                                .addStyle("border-right", "2px solid " + Colors.WHITE.getValue())
                );
            }

            if (removed > 0) {
                titles.add(
                        tdRight(
                                bDot(Colors.SEARCHABLE_REMOVED.getValue()),
                                HtmlNode.safeText(i18n.getSearchableRemoved(DigestTextUtil.formatNumber(stats.getRemoved().getView())))
                        ).addStyle(Styles.B_SEARCHABLE_BAR_STATS)
                );

                strips.add(
                        HtmlNode.td(NBSP)
                                .addStyle(Styles.B_SEARCHABLE_BAR_STRIP)
                                .addStyle("background-color", Colors.SEARCHABLE_REMOVED.getValue())
                                .addStyle("width", String.valueOf(removedPercent) + "%")
                );
            }

            return HtmlNode.div()
                    .addStyle("margin", "0 0 15px")
                    .addContent(
                            bTable().addRow(titles),
                            bTable().addRow(strips)
                    );
        }
    }

    public class ImportantPagesBlocks {
        public final String UTM_TERM_IMPORTANT_PAGES = "important_pages";

        public HtmlNode bImportant(Optional<DigestData.ImportantUrls> importantUrlsOpt) {
            return importantUrlsOpt.map(importantUrls -> bSection(
                    bTable()
                            .addHeader(
                                    thLeft(tableTitle(HtmlNode.safeText(i18n.getImportantPagesText())))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6))),
                                    thRight(HtmlNode.safeText(i18n.getImportantCodeText()))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2))),
                                    thRight(HtmlNode.safeText(i18n.getImportantStatusText()))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2)))
                            )
                            .addRows(
                                    importantUrls.getImportantUrls().stream().map(this::bImportantPage).collect(Collectors.toList())
                            ),
                    bLandingLink(
                            HtmlNode.safeText(i18n.getImportantPagesLanding()),
                            webmasterLinksProvider.importantUrls(host.getId(), false, UTM_TERM_IMPORTANT_PAGES)
                    )
            )).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode.TrDTag bImportantPage(DigestData.ImportantUrl link) {
            return HtmlNode.trd(
                    tdLeft(
                            link.getTitle().isEmpty() ? HtmlNode.EMPTY :
                                    HtmlNode.p(HtmlNode.text(link.getTitle()))
                                            .addStyle(Styles.SEARCHABLE_TITLE)
                                            .addStyle(Styles.WRAP_BREAK_WORDS),
                            link.getUrl().isEmpty() ? HtmlNode.EMPTY :
                                    HtmlNode.p(HtmlNode.text(link.getUrl()))
                                            .addStyle(Styles.SEARCHABLE_URL)
                                            .addStyle(Styles.WRAP_ELLIPSIS)
                                            .styleWidth(HtmlSize.px((int)(MAIL_WIDTH * 0.6)))
                    ),
                    tdRight(highlight(link.getIndexingHttpCode())),
                    tdRight(highlight(link.getSearchUrlStatus()))
            );
        }
    }

    public class RecommendedUrlsBlocks {
        public final String UTM_TERM_RECOMMENDED_IMPORTANT_PAGES = "recommended_important_pages";

        public HtmlNode bRecommended( Optional<DigestData.RecommendedUrls> recommendedUrl) {
            return recommendedUrl
                    .filter(recommendedUrls -> recommendedUrls.getRecommendedUrls() != null && !recommendedUrls.getRecommendedUrls().isEmpty())
                    .map(recommendedUrls -> bSection(
                            bTable()
                                    .addHeader(
                                            thLeft(tableTitle(HtmlNode.safeText(i18n.getRecommendedPagesText())))
                                                    .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6)))
                                                    .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6)))
                                                    .addStyle(Styles.WRAP_NOWRAP)
                                            ,
                                            thRight(HtmlNode.safeText(i18n.getRecommendedShowsText()))
                                                    .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2))),
                                            thRight(HtmlNode.safeText(i18n.getRecommendedClicksText()))
                                                    .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2)))
                                    )
                                    .addRows(
                                            recommendedUrls.getRecommendedUrls().stream().limit(3).map(this::bRecommendedPage).collect(Collectors.toList())
                                    ),
                            bLandingLink(
                                    HtmlNode.safeText(recommendedUrls.getRecommendedUrls().size() < 4 ? i18n.getRecommendedPagesLandingFew() : i18n.getRecommendedPagesLanding()),
                                    webmasterLinksProvider.importantUrls(host.getId(), true, UTM_TERM_RECOMMENDED_IMPORTANT_PAGES)
                            )
                    )).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode.TrDTag bRecommendedPage(DigestData.RecommendedUrl link) {
            return HtmlNode.trd(
                    tdLeft(
                            HtmlNode.text(URLEncodeUtil.urlDecodeSafe(link.getUrl()))
                    )
                            .addStyle(Styles.WRAP_BREAK_WORDS)
                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6)))
                    ,
                    tdRight(bNumber(link.getShows())).styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2))),
                    tdRight(bNumber(link.getClicks()))).styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2)));
        }
    }

    public HtmlNode highlight(DigestData.ImportantUrl.HighlightedValue value) {
        HtmlNode text = HtmlNode.text(value.getValue());
        if (value.isHighlight()) {
            return bChanged(text);
        }
        return text;
    }

    /* График кликов */
    public class ClicksChartBlocks {
        public final String UTM_TERM_CLICKS_HISTORY = "clicks_history";

        public HtmlNode bClicksChart(Optional<DigestData.Clicks> clicks, LocalDate prevWeekFrom, LocalDate prevWeekTo) {
            if (!attaches.contains(DigestAttachType.CLICKS_DYNAMICS) || clicks.isEmpty()) {
                return HtmlNode.EMPTY;
            }
            return bSection(
                    bSectionTitle(I18nDigestClicks.HTML_title.newBuilder(i18n.language)
                            .count(clicks.get().getCount(), DigestBlocksBuilder::bNumber)).addStyle("margin", "0"),
                    bTable().addRow(
                            bClicksDifference(clicks.get().getPastWeekylClicksCount(), clicks.get().getCount()),
                            tdRight(
                                    bLegendPast(clicks.get().getPastWeekylClicksCount(),
                                                    prevWeekFrom, prevWeekTo),
                                    HtmlNode.span()
                                            .addStyle(Styles.SMALL_TEXT)
                                            .addStyle("margin-left", "15px")
                                            .addContent(
                                                    bDotLegend(chartSettings.getCurrentSeriesColor()),
                                                    HtmlNode.text(DigestTextUtil.shortDateRange(i18n.language, digestFrom, digestTo)))
                            ).addStyle(Styles.WRAP_NOWRAP)
                                    .addStyle(Styles.CLICKS_CHART_LEGEND)
                    ),
                    HtmlNode.a(webmasterLinksProvider.queryGroupClickStats(
                            host.getId(),
                            prevWeekFrom,
                            digestTo,
                            QueryIndicator.TOTAL_CLICKS_COUNT,
                            UTM_TERM_CLICKS_HISTORY
                    )).addContent(
                            new HtmlNode.ImgTag(getCidURL(DigestAttachType.CLICKS_DYNAMICS))
                                    .width(HtmlSize.px(480))
                                    .height(HtmlSize.px(180))
                                    .addStyle("background", Colors.GRAY.getValue())
                                    .addStyle("border", "none")
                    ).addStyle("text-decoration", "none"));
        }

        private HtmlNode bLegendPast(Optional<Long> past, LocalDate prevWeekFrom, LocalDate prevWeekTo) {
            if (!past.isPresent()) {
                return HtmlNode.EMPTY;
            }
            return HtmlNode.span()
                    .addStyle(Styles.SMALL_TEXT)
                    .addContent(
                            bDotLegend(chartSettings.getPastSeriesColor()),
                            HtmlNode.text(DigestTextUtil.shortDateRange(i18n.language, prevWeekFrom, prevWeekTo)));
        }

        private HtmlNode.TdTag bClicksDifference(Optional<Long> past, Long current) {
            if (!past.isPresent()) {
                return tdLeft(HtmlNode.EMPTY);
            }
            return tdLeft(HtmlNode.text(i18n.getClicksDifference(Long.signum(current - past.get()),
                    DigestTextUtil.formatNumber((Math.abs(current - past.get()))))))
                    .addStyle("color", Colors.GRAY.getValue())
                    .addStyle("padding", "4px 0 16px");
        }

    }

    public class QueriesBlocks {
        public final String UTM_TERM_SEARCH_QUERY_STAT = "search_query_stat";

        public HtmlNode bQueries(Optional<DigestData.Queries> queriesOpt) {
            return queriesOpt.map(queries -> bSection(
                        bTable().addHeader(
                                thLeft(tableTitle(HtmlNode.safeText(i18n.getQueriesTitle())))
                                        .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6))),
                                thRight(HtmlNode.text(i18n.getQueriesTableClicks()))
                                        .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2))),
                                thRight(HtmlNode.text(i18n.getQueriesTableDifference()))
                                        .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2)))
                                ).addRows(
                                        queries.getData().stream().map(this::bQueriesRow).collect(Collectors.toList())
                        ),
                        bLandingLink(
                                HtmlNode.safeText(i18n.getQueriesLink()),
                                webmasterLinksProvider.queriesClickStats(host.getId(), digestFrom, digestTo, UTM_TERM_SEARCH_QUERY_STAT)
                        )
            )).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode.TrDTag bQueriesRow(DigestData.Queries.QData query) {
            return HtmlNode.trd(
                    tdLeft(HtmlNode.text(query.getQuery()))
                            .addStyle(Styles.WRAP_BREAK_WORDS),
                    tdRight(bNumber(query.getClicks()))
                            .addStyle(Styles.WRAP_NOWRAP),
                    tdRight(bGrowth(query.getDifference(),false))
                            .addStyle(Styles.WRAP_NOWRAP)
            );
        }
    }

    public class ReviewsBlock {
        public final String UTM_TERM_REVIEWS_BLOCK = "digest_reviews";

        public HtmlNode bReviews(Optional<DigestData.Reviews> reviewsOptional) {
            if (reviewsOptional.isEmpty()){
                return HtmlNode.EMPTY;
            }
            DigestData.Reviews reviews = reviewsOptional.get();
            return bSection(
                    bSectionTitle(
                            HtmlNode.text(
                                    I18nDigestReviewTexts.reviews_title.getText(i18n.language)
                            )
                    ),
                    HtmlNode.p(
                            I18nDigestReviewTexts.HTML_reviews_text.newBuilder(i18n.language)
                                    .count(reviews.getDelta(), DigestBlocksBuilder::bNumber)
                                    .link(
                                            HtmlNode.a(webmasterLinksProvider.reviewsLink(host.getId(), UTM_TERM_REVIEWS_BLOCK))
                                                    .addStyle(Styles.LANDING_LINK)
                                                    .addContent(HtmlNode.text(I18nDigestReviewTexts.reviews_link.getText(i18n.language)))
                                    )
                    )
            );
        }
    }


    public class MetrikaCounterProblemBlock {
        public final String UTM_TERM_METRIKA_COUNTER_BLOCK = "digest_metrika";

        public HtmlNode bMetrikaCounterProblem(Optional<DigestData.MetrikaCounterProblem> counterProblemOpt) {
            if (!isFullDigest() || counterProblemOpt.isEmpty()){
                return HtmlNode.EMPTY;
            }

            SiteProblemTypeEnum problemType = counterProblemOpt.get().getProblemType();
            return bSection(
                    getTitle(problemType),
                    getText(problemType),
                    getActionLink(problemType)
            );
        }

        private HtmlNode getTitle(SiteProblemTypeEnum problemType) {
            switch (problemType) {
                case NO_METRIKA_COUNTER_BINDING:
                    return getTitleSection(
                            HtmlNode.text(
                                    I18nDigestMetrikaCounterProblem.no_metrika_counter_binding_title.getText(i18n.language)
                            )
                    );

                case NO_METRIKA_COUNTER_CRAWL_ENABLED:
                    return getTitleSection(
                            HtmlNode.text(
                                    I18nDigestMetrikaCounterProblem.no_metrika_counter_crawl_enabled_title.getText(i18n.language)
                            )
                    );

                default:
                    return HtmlNode.EMPTY;
            }
        }

        private HtmlNode getTitleSection(HtmlNode title) {
            return HtmlNode.h3(title).addStyle(Styles.SECTION_SMALL_TITLE);
        }

        private HtmlNode getText(SiteProblemTypeEnum problemType) {
            switch (problemType) {
                case NO_METRIKA_COUNTER_BINDING:
                    return HtmlNode.div()
                            .addContent(
                                    HtmlNode.p(
                                            I18nDigestMetrikaCounterProblem.HTML_no_metrika_counter_binding_text.newBuilder(i18n.language)
                                                    .metrika_counter_help_link(
                                                            HtmlNode.a(webmasterLinksProvider.supportLinkMetrika())
                                                                    .addStyle(Styles.LANDING_LINK)
                                                                    .addContent(HtmlNode.text(I18nDigestMetrikaCounterProblem.metrika_counter_help_link.getText(i18n.language)))
                                                    )
                                    )
                            );

                case NO_METRIKA_COUNTER_CRAWL_ENABLED:
                    return HtmlNode.div()
                            .addContent(
                                    HtmlNode.p(
                                            I18nDigestMetrikaCounterProblem.HTML_no_metrika_counter_crawl_enabled_text.newBuilder(i18n.language)
                                                    .metrika_counter_help_link(
                                                            HtmlNode.a(webmasterLinksProvider.supportLinkMetrika())
                                                                    .addStyle(Styles.LANDING_LINK)
                                                                    .addContent(HtmlNode.text(I18nDigestMetrikaCounterProblem.metrika_counter_help_link.getText(i18n.language)))
                                                    )
                                    )
                            );

                default:
                    return HtmlNode.EMPTY;
            }
        }

        private HtmlNode getActionLink(SiteProblemTypeEnum problemType) {
            switch (problemType) {
                case NO_METRIKA_COUNTER_BINDING:
                    return bLandingLink(
                            HtmlNode.safeText(I18nDigestMetrikaCounterProblem.no_metrika_counter_binding_link.getText(i18n.language)),
                            webmasterLinksProvider.hostMetrikaSettings(host.getId(), UTM_TERM_METRIKA_COUNTER_BLOCK)
                    );

                case NO_METRIKA_COUNTER_CRAWL_ENABLED:
                    return bLandingLink(
                            HtmlNode.safeText(I18nDigestMetrikaCounterProblem.no_metrika_counter_crawl_enabled_link.getText(i18n.language)),
                            webmasterLinksProvider.hostIndexingCrawlMetrika(host.getId(), UTM_TERM_METRIKA_COUNTER_BLOCK)
                    );

                default:
                    return HtmlNode.EMPTY;
            }
        }
    }

    public class QueriesGroupsBlocks {
        public final String UTM_TERM_QUERY_GROUP_STAT = "query_group_stat";

        public HtmlNode bQueriesGroups(Optional<DigestData.QueriesGroups> queriesGroupsOpt) {
            return queriesGroupsOpt.map(queriesGroups -> bSection(
                    bTable()
                            .addHeader(
                                    thLeft(tableTitle(HtmlNode.safeText(i18n.getQueriesGroupsTitle())))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.6))),
                                    thRight(HtmlNode.safeText(i18n.getQueriesGroupsTableClicks()))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2))),
                                    thRight(HtmlNode.safeText(i18n.getQueriesGroupsTableDifference()))
                                            .styleWidth(HtmlSize.px((int) (MAIL_WIDTH * 0.2)))
                            ).addRows(queriesGroups.getData().stream().map(this::bQueriesGroupRow).collect(Collectors.toList())),
                    bLandingLink(
                            HtmlNode.safeText(i18n.getQueriesGroupsLink()),
                            webmasterLinksProvider.queryGroupClickStats(
                                    host.getId(),
                                    digestFrom,
                                    digestTo,
                                    QueryIndicator.CLICKS_DYNAMICS,
                                    UTM_TERM_QUERY_GROUP_STAT
                            )
                    )
            )).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode.TrDTag bQueriesGroupRow(DigestData.QueriesGroups.QDataGroups group) {
            return HtmlNode.trd(
                    tdLeft(HtmlNode.text(group.getTitle()))
                            .addStyle(Styles.WRAP_BREAK_WORDS),
                    tdRight(bNumber(group.getClicks()))
                            .addStyle(Styles.WRAP_NOWRAP),
                    tdRight(bGrowth(group.getDifference(),false))
                            .addStyle(Styles.WRAP_NOWRAP)
            );
        }
    }

    public HtmlNode bGrowth(DigestData.Difference number, boolean percent) {
        if (number == null || (percent && number.getValue() == 0)) {
            return HtmlNode.EMPTY;
        }

        return HtmlNode.span()
                .addStyle("color", (Long.signum(number.getValue()) < 0) ? Colors.RED.getValue() : Colors.GREEN.getValue())
                .addContent(bNumber(Math.abs(number.getValue())))
                .addContent(percent ? HtmlNode.text("%") : HtmlNode.EMPTY)
                .addContent(
                        new HtmlNode.BTag()
                                .addStyle(Styles.ARROW)
                                .addContent(HtmlNode.text((Long.signum(number.getValue()) < 0) ? "↓" : "↑"))
                );
    }

    public HtmlNode bLandingLink(HtmlNode content, String url) {
        if (url == null || url.isEmpty()) {
            return HtmlNode.EMPTY;
        }
        return HtmlNode.p()
                .addStyle(Styles.LANDING_PARAGRAPH)
                .addContent(
                        HtmlNode.a(url)
                                .addStyle(Styles.LANDING_LINK)
                                .addContent(content)
                );
    }

    public HtmlNode.HTag bSectionTitle(HtmlNode title) {
        return HtmlNode.h2(title)
                .addStyle(Styles.SECTION_TITLE);
    }

    /* Точечки */
    public HtmlNode bDot(String color) {
        return HtmlNode.span()
                .addStyle(Styles.DOT)
                .addStyle("color", color)
                .addContent(BLACK_CIRCLE);
    }

    /* Точечки на легенде графика кликов*/
    public HtmlNode bDotLegend(String color) {
        return HtmlNode.span()
                .addStyle(Styles.DOT_LEGEND)
                .addStyle("color", color)
                .addContent(BLACK_CIRCLE);
    }

    public HtmlNode.ThTag thLeft(HtmlNode... content) {
        return HtmlNode.th(content)
                .addStyle(Styles.TH)
                .addStyle(Styles.TH_LEFT);
    }

    public HtmlNode.ThTag thRight(HtmlNode... content) {
        return HtmlNode.th(content)
                .addStyle(Styles.TH)
                .addStyle(Styles.TH_RIGHT);
    }

    public HtmlNode.HTag tableTitle(HtmlNode... content) {
        return HtmlNode.h2(content)
                .addStyle(Styles.TABLE_TITLE);
    }

    public HtmlNode.TdTag tdLeft(HtmlNode... content) {
        return new HtmlNode.TdTag()
                .addStyle(Styles.TD)
                .addStyle(Styles.TD_LEFT)
                .addContent(content);
    }

    public HtmlNode.TdTag tdRight(HtmlNode... content) {
        return new HtmlNode.TdTag()
                .addStyle(Styles.TD)
                .addStyle(Styles.TD_RIGHT)
                .addContent(content);
    }

    public class BlogPostsBlocks {
        private final String BLOG_URL = "https://webmaster.yandex.ru/blog/";

        public HtmlNode bBlogPosts(Optional<DigestData.BlogPosts> blogPostsOpt) {
            return blogPostsOpt.map(blogPosts -> bSection(
                bSectionTitle(HtmlNode.text(i18n.getBlogTitle())),

                HtmlNode.list(
                       blogPosts.getBlogPosts()
                            .stream()
                            .map(this::bBlogPost)
                            .collect(Collectors.toList())
                    )

            )).orElse(HtmlNode.EMPTY);
        }

        public HtmlNode bBlogPost(DigestData.BlogPost post) {
            return HtmlNode.a(webmasterLinksProvider.makeLinkToPost(BLOG_URL, post.getSlug()))
                .addStyle(Styles.LANDING_LINK)
                .addContent(
                    HtmlNode.div()
                        .addContent(
                            HtmlNode.p()
                                .addStyle(Styles.BLOG_POST_TITLE)
                                .addContent(HtmlNode.text(post.getTitle())),
                            HtmlNode.p()
                                .addStyle(Styles.BLOG_POST_DATE)
                                .addContent(HtmlNode.text(DigestTextUtil.dateInPost(i18n.language, post.getDate())))
                        )
                );
        }
    }

    public HtmlNode bFooter() {
        return HtmlNode.div()
                .addContent(
                        HtmlNode.p(
                                i18n.getFooterViewChanges(
                                        HtmlNode.a(webmasterLinksProvider.hostDashboard(host.getId(), UTM_TERM_DASHBOARD_FOOTER))
                                                .addStyle(Styles.LANDING_LINK)
                                                .addContent(HtmlNode.safeText(i18n.getFooterInSummary()))
                                )
                        ).addStyle(Styles.FOOTER),
                        HtmlNode.p(
                                i18n.getFooterManageSubscriptions(
                                        HtmlNode.a(webmasterLinksProvider.notificationSettings(UTM_TERM_UNSUBSCRIBE))
                                                .addStyle(Styles.LANDING_LINK)
                                                .addContent(HtmlNode.safeText(i18n.getFooterInSettings()))
                                )

                        ).addStyle(Styles.FOOTER),
                        HtmlNode.p(HtmlNode.text(i18n.getFooterEnd())).addStyle(Styles.FOOTER)
                );
    }

    public HtmlNode bChanged(HtmlNode... content) {
        return new HtmlNode.SpanTag()
                .addStyle(Styles.CHANGED)
                .addContent(content);
    }

    private boolean isFullDigest(){
        return notificationType == NotificationType.DIGEST;
    }

    private static String getCidURL(DigestAttachType attachType){
        return "cid:" + attachType.getCid();
    }

    public enum Colors {
        WHITE("#fff"),
        BLACK("#000"),
        LINK("#0044bb"),
        RED("#ff3333"),
        GREEN("#32d563"),
        GRAY("#858e98"),
        SEARCHABLE_ADDED("#52db4a"),
        SEARCHABLE_REMOVED("#2c91f5"),;

        private final String value;

        Colors(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    public enum Fonts {
        MAIN("Arial, Helvetica, sans-serif"),
        GRAPHICS("Verdana, sans-serif"),;

        private final String value;

        Fonts(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    private static class Styles {
        static final CSS CONTAINER = CSS.create()
                .set("line-height", "1")
                .set("font-family", Fonts.MAIN.getValue())
                .set("background", Colors.WHITE.getValue())
                .set("text-align", "left")
                .set("font-size", "14px")
                .set("color", Colors.BLACK.getValue())
                .set("width", "100%");

        static final CSS TABLE = CSS.create()
                .set("width", "100%")
                .set("border-collapse", "collapse")
                .set("font-size", "14px")
                .set("line-height", "1")
                .set("margin", "0")
                .set("font-family", "Arial, Helvetica, sans-serif")
                .set("table-layout", "fixed");

        static final CSS SMALL_TEXT = CSS.create()
                .set("font-size", "10px")
                .set("line-height", "1.5")
                .set("color", Colors.GRAY.getValue());

        static final CSS MAIN_TABLE = CSS.create()
                .set("padding", "24px 18px")
                .set("width", String.format("%dpx", MAIL_WIDTH));

        /**
         * Шапка письма (заголовок и даты)
         */
        private static final CSS MAIN_HEADER = CSS.create()
                .set("margin", "0 0 26px");

        private static final CSS MAIN_TITLE = CSS.create()
                .set("font-size", "28px")
                .set("font-weight", "bold")
                .set("margin", "0 0 7px");

        private static final CSS MAIN_HOST_LINK = CSS.create()
                .set("color", Colors.LINK.getValue())
                .set("text-decoration", "none");

        private static final CSS MAIN_PERIOD = CSS.create()
                .set("font-size", "15px")
                .set("color", Colors.GRAY.getValue())
                .set("margin", "0");

        private static final CSS LANDING_PARAGRAPH = CSS.create()
                .set("margin", "10px 0 0")
                .set("font-size", "14px");

        private static final CSS LANDING_LINK = CSS.create()
                .set("font-weight", "bold")
                .set("color", Colors.LINK.getValue())
                .set("text-decoration", "none");

        /* Секции */
        private static final CSS SECTION = CSS.create()
                .set("margin", "25px 0 30px");

        /* Заголовки секций */
        private static final CSS SECTION_TITLE = CSS.create()
                .set("font-family", Fonts.MAIN.getValue())
                .set("font-size", "18px")
                .set("font-weight", "bold")
                .set("line-height", "1.2")
                .set("margin", "0 0 10px")
                .set("color", Colors.BLACK.getValue());

        private static final CSS SECTION_SMALL_TITLE = CSS.create()
                .set("font-family", Fonts.MAIN.getValue())
                .set("font-size", "16px")
                .set("font-weight", "bold")
                .set("line-height", "1.2")
                .set("margin", "0 0 10px")
                .set("color", Colors.BLACK.getValue());

        /* Чеклист */
        private static final CSS CHECKLIST_CHUNK = CSS.create()
                .set("margin", "0 0 30px");

        private static final CSS CHECKLIST_LIST = CSS.create()
                .set("margin", "2px 0 16px")
                .set("padding", "0");

        private static final CSS CHECKLIST_ITEM = CSS.create()
                .set("list-style", "none")
                .set("margin", "0 0 10px");

        /* Стили про перенос слов */
        /* Переносить слово, если оно не входит в строку */
        private static final CSS WRAP_BREAK_WORDS = CSS.create()
                .set("word-wrap", "break-word")
                .set("word-break", "break-all");

        /* Показывать всё в одну строку и прятать под многоточие */
        private static final CSS WRAP_ELLIPSIS = CSS.create()
                .set("white-space", "nowrap")
                .set("overflow", "hidden")
                .set("text-overflow", "ellipsis");

        /* Показывать в одну строку и не прятать */
        private static final CSS WRAP_NOWRAP = CSS.create()
                .set("white-space", "nowrap");

        /* Линия-разделитель */
        private static final CSS DELIMITER = CSS.create()
                .set("height", "1px")
                .set("border", "none")
                .set("background-color", "#e1e1e1");

        private static final CSS B_SEARCHABLE_BAR_STATS = CSS.create()
                .set("padding", "0 0 7px");

        private static final CSS B_SEARCHABLE_BAR_STRIP = CSS.create()
                .set("border", "none")
                .set("font-size", "0")
                .set("height", "6px")
                .set("padding", "0");

        private static final CSS TD = CSS.create()
                .set("vertical-align", "baseline");

        private static final CSS TD_LEFT = CSS.create()
                .set("text-align", "left")
                .set("padding", "8px 4px 8px 0");

        private static final CSS TD_RIGHT = CSS.create()
                .set("text-align", "right")
                .set("padding", "8px 0 8px 4px");

        private static final CSS TH = CSS.create()
                .set("font-weight", "normal")
                .set("text-transform", "uppercase")
                .set("font-size", "10px")
                .set("line-height", "1.5")
                .set("letter-spacing", "0.5px")
                .set("color", Colors.GRAY.getValue())
                .set("padding", "0 0 8px")
                .set("vertical-align", "baseline");

        private static final CSS TH_LEFT = CSS.create()
                .set("text-align", "left");

        private static final CSS TH_RIGHT = CSS.create()
                .set("text-align", "right");

        private static final CSS TABLE_TITLE = CSS.create()
                .set("font-family", Fonts.MAIN.getValue())
                .set("font-size", "18px")
                .set("font-weight", "bold")
                .set("line-height", "1.33")
                .set("text-transform", "none")
                .set("letter-spacing", "normal")
                .set("margin", "0")
                .set("color", Colors.BLACK.getValue());

        private static final CSS SEARCHABLE_TITLE = CSS.create()
                .set("margin", "0 0 5px");

        private static final CSS SEARCHABLE_URL = CSS.create()
                .set("margin", "0 0 4px")
                .set("line-height", "1.2")
                .set("color", Colors.GRAY.getValue());

        private static final CSS SEARCHABLE_REASON = CSS.create()
                .set("color", Colors.GRAY.getValue());

        private static final CSS SEARCHABLE_STATUS_ADDED = CSS.create()
                .set("color", Colors.SEARCHABLE_ADDED.getValue());

        private static final CSS SEARCHABLE_STATUS_REMOVED = CSS.create()
                .set("color", Colors.SEARCHABLE_REMOVED.getValue());

        private static CSS getSearchable(DigestData.StatsEnum status) {
            if (status == DigestData.StatsEnum.ADDED) {
                return SEARCHABLE_STATUS_ADDED;
            } else {
                return SEARCHABLE_STATUS_REMOVED;
            }
        }

        private static final CSS ARROW = CSS.create()
                .set("font-family", Fonts.GRAPHICS.getValue())
                .set("font-weight", "bold")
                .set("margin-left", "8px");

        private static final CSS CLICKS_CHART_LEGEND = CSS.create()
                .set("padding", "4px 0 16px");

        private static final CSS CHANGED = CSS.create()
                .set("background", "#ffeba0")
                .set("padding", "2px 6px")
                .set("margin", "0 -6px");

        private static final CSS STYLES_TD = CSS.create()
                .set("vertical-align", "baseline");

        private static final CSS FOOTER = CSS.create()
                .set("margin", "14px 0")
                .set("font-size", "12px");

        private static final CSS DOT = CSS.create()
                .set("font-size", "16px")
                .set("margin-right", "4px")
                .set("vertical-align", "baseline");

        private static final CSS DOT_LEGEND = CSS.create()
                .set("font-size", "16px")
                .set("margin-right", "0")
                .set("vertical-align", "-1px");

        private static final CSS LOGO = CSS.create()
                .add(STYLES_TD)
                .set("padding", "0 0 16px");

        private static final CSS BLOG_POST_TITLE = CSS.create()
                .set("margin", "0 0 3px");

        private static final CSS BLOG_POST_DATE = CSS.create()
                .set("color", Colors.GRAY.getValue())
                .set("font-weight", "normal")
                .set("font-size", "13px")
                .set("margin", "0 0 10px");
    }
}
