package ru.yandex.webmaster3.worker.digest.html;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import NWebmaster.proto.digest.Digest;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.digest.DigestChecklistUtil;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.searchquery.SpecialGroup;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil;
import ru.yandex.webmaster3.storage.achievements.model.AchievementTld;
import ru.yandex.webmaster3.storage.achievements.model.AchievementType;
import ru.yandex.webmaster3.storage.achievements.model.official.AchievementOfficialType;
import ru.yandex.webmaster3.storage.user.dao.PersonalInfoCacheYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;

/**
 * Created by ifilippov5 on 28.09.17.
 */
public class DigestDataBuilder {
    private static final Logger log = LoggerFactory.getLogger(DigestDataBuilder.class);
    private DigestData.Host host;
    private Optional<DigestData.Checklist> checklist;
    private Optional<DigestData.Searchable> searchable;
    private Optional<DigestData.Clicks> clicks;
    private Optional<DigestData.Queries> queries;
    private Optional<DigestData.QueriesGroups> queriesGroups;
    private Optional<List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>>> achievements;
    private Optional<DigestData.ImportantUrls> importantUrl;
    private Optional<DigestData.RecommendedUrls> recommendedUrl;
    private LocalDate dateFrom;
    private LocalDate dateTo;
    private Optional<DigestData.BlogPosts> blogPosts;
    private Optional<DigestData.Reviews> reviews;
    private Optional<DigestData.Mirrors> mirrors;
    private Optional<DigestData.MetrikaCounterProblem> metrikaCounterProblem;
    private Optional<Boolean> isFirstLiteDigest;

    public static int IMPORTANT_URLS_QUOTA = 100;

    private final static int ACHIEVEMENT_MAX_STAR = 5;

    private DigestI18n i18n;

    public DigestDataBuilder(DigestI18n i18n) {
        this.i18n = i18n;
    }

    public void withHost(WebmasterHostId hostId) {
        this.host = new DigestData.Host(hostId, DigestTextUtil.readableHost(hostId));
    }

    public void withPeriod(LocalDate dateFrom, LocalDate dateTo) {
        this.dateFrom = dateFrom;
        this.dateTo = dateTo;
    }

    //фатальные-критичные-метрика
    private static Comparator<SiteProblemTypeEnum> getProblemsComparator() {
        return Comparator.comparing(p -> p.getSeverity().value());
    }

    public void withChecklist(List<Integer> problemListOld,
                              List<Integer> problemListNew,

                              List<Digest.GeoRegionInfo> regionsListOld,
                              List<Digest.GeoRegionInfo> regionsListNew,

                              String oldRegister,
                              String newRegister,

                              int recrawledCount,

                              List<Digest.UserVerificationState> userVerificationStateList,
                              PersonalInfoCacheYDao personalInfoCacheYDao,

                              boolean isYmlMarketTeaser,
                              boolean hasOffAutoparser,
                              long turboAutoparsedPagesCount,

                              Digest.IKSState oldIks,
                              Digest.IKSState newIks,
                              String owner
    ) {

        Set<SiteProblemTypeEnum> oldProblemsSet = problemListOld.stream()
                .flatMap(p -> {
                    SiteProblemTypeEnum problemOpt = SiteProblemTypeEnum.R.fromValueOrNull(p);
                    return problemOpt == null ? Stream.empty() : Stream.of(problemOpt);
                })
                .collect(Collectors.toCollection(() -> EnumSet.noneOf(SiteProblemTypeEnum.class)));
        Set<SiteProblemTypeEnum> newProblemsSet = problemListNew.stream()
                .flatMap(p -> {
                    SiteProblemTypeEnum problemOpt = SiteProblemTypeEnum.R.fromValueOrNull(p);
                    return problemOpt == null ? Stream.empty() : Stream.of(problemOpt);
                })
                .collect(Collectors.toCollection(() -> EnumSet.noneOf(SiteProblemTypeEnum.class)));

        List<SiteProblemTypeEnum> problems = newProblemsSet
                .stream()
                .filter(DigestChecklistUtil::showProblemIfPresent)
                .sorted(getProblemsComparator())
                .collect(Collectors.toList());
        MutableBoolean problemsOnlyInTurbo = new MutableBoolean(!problems.isEmpty());
        MutableBoolean problemsOnlyInTurboContent = new MutableBoolean(!problems.isEmpty());

        problems.forEach(problemType -> problemsOnlyInTurbo.setValue(problemsOnlyInTurbo.booleanValue() && problemType.isTurboProblem()));
        problems.forEach(problemType -> problemsOnlyInTurboContent.setValue(problemsOnlyInTurboContent.booleanValue() && problemType.isTurboContentProblem(true)));

        Set<SiteProblemTypeEnum> fixedProblems = EnumSet.copyOf(oldProblemsSet);
        fixedProblems.removeAll(newProblemsSet);
        List<SiteProblemTypeEnum> problemsResolved = fixedProblems
                .stream()
                .filter(DigestChecklistUtil::showProblemIfFixed)
                .sorted(getProblemsComparator())
                .collect(Collectors.toList());

        List<DigestData.HostEvent> hostEvents = new ArrayList<>();

        if (newIks != null) {
            if (oldIks == null || oldIks.getTimestamp() != newIks.getTimestamp()) {
                hostEvents.add(new DigestData.IksUpdated(newIks.getIks(), owner));
            }
        }

        if (!regionsEquals(regionsListOld, regionsListNew)) {
            hostEvents.add(new DigestData.RegionChanged());
        }

        if (!urlWithoutSchema(oldRegister).equals(urlWithoutSchema(newRegister))) {
            hostEvents.add(new DigestData.HostDisplayNameChanged());
        }

        if (isYmlMarketTeaser) {
            hostEvents.add(new DigestData.YmlTurboAvailable(host.getId()));
        }

//        if (recrawledCount > 0) {
//            hostEvents.add(new DigestData.RecrawledPages(recrawledCount));
//        }

        try {
            hostEvents.addAll(personalInfoCacheYDao.getLogins(userVerificationStateList
                    .stream()
                    .map(Digest.UserVerificationState::getUserId)
                    .collect(Collectors.toList())
            )
                    .stream()
                    .map(DigestData.UserVerifiedHost::new)
                    .collect(Collectors.toList()));
        } catch (WebmasterYdbException e) {
            //ignore
            log.error("Unable to get logins", e);
        }
        if (hasOffAutoparser && (turboAutoparsedPagesCount > 1)) {
            hostEvents.add(new DigestData.AutoparserOff(turboAutoparsedPagesCount));
        }
        boolean countNotEmpty = !problems.isEmpty() || !problemsResolved.isEmpty() || !hostEvents.isEmpty();
        DigestData.Checklist checklist = countNotEmpty ?
                new DigestData.Checklist(problems, problemsOnlyInTurbo.booleanValue(), problemsOnlyInTurboContent.booleanValue(), problemsResolved, hostEvents) : null;
        this.checklist = Optional.ofNullable(checklist);
    }

    private List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>> createAchievements(
            List<Digest.AchievementsSnapshot> achievementsOld,
            List<Digest.AchievementsSnapshot> achievementsNew
    ) {
        Map<AchievementTld, Digest.AchievementsSnapshot> mapOld = achievementsOld.stream()
                .collect(Collectors.toMap(
                        a -> AchievementTld.getIfPresent(StringUtils.upperCase(a.getTld())),
                        Function.identity()));
        Map<AchievementTld, Digest.AchievementsSnapshot> mapNew = achievementsNew.stream()
                .collect(Collectors.toMap(
                        a -> AchievementTld.getIfPresent(StringUtils.upperCase(a.getTld())),
                        Function.identity()));

        List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>> tldAchievements = new ArrayList<>();
        mapOld.forEach((k, v) -> {
            if (mapNew.containsKey(k)) {
                tldAchievements.add(createAchievements(v, mapNew.get(k), k));
            }
        });

        return tldAchievements;
    }


    private Pair<AchievementTld, List<DigestData.AbstractAchievement>> createAchievements(
            Digest.AchievementsSnapshot achievementsOld,
            Digest.AchievementsSnapshot achievementsNew,
            AchievementTld tld
    ) {

        List<DigestData.AbstractAchievement> achievements = new ArrayList<>();

        if (!achievementsOld.getOfficial() && achievementsNew.getOfficial()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.GOV));
        }

        String oldAuto = achievementsOld.hasOfficialAuto() ? achievementsOld.getOfficialAuto() : null;
        String newAuto = achievementsNew.hasOfficialAuto() ? achievementsNew.getOfficialAuto() : null;

        if (!StringUtils.equals(oldAuto, newAuto) && !StringUtils.isEmpty(newAuto)) {
            achievements.add(new DigestData.BrandedOfficialAchievement(tld, AchievementOfficialType.AUTO, newAuto));
        }

        if (!achievementsOld.getOfficialAvia() && achievementsNew.getOfficialAvia()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.AIRLINE));
        }

        if (!achievementsOld.getOfficialCbr() && achievementsNew.getOfficialCbr()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.CREDIT_ORGANIZATION));
        }

        if (!achievementsOld.getOfficialEmbassy() && achievementsNew.getOfficialEmbassy()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.EMBASSY));
        }

        if (!achievementsOld.getOfficialMfo() && achievementsNew.getOfficialMfo()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.MFO));
        }

        if (!achievementsOld.getOfficialSsd() && achievementsNew.getOfficialSsd()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.SSD));
        }

        if (!achievementsOld.getOfficialVisaCenter() && achievementsNew.getOfficialVisaCenter()) {
            achievements.add(new DigestData.AchievementOfficial(tld, AchievementOfficialType.VISA_CENTER));
        }

        if (!achievementsOld.getTurbo() && achievementsNew.getTurbo()) {
            achievements.add(new DigestData.Achievement(tld, AchievementType.TURBO));
        }

        if ((achievementsNew.hasSpeedGradeMobile() && achievementsNew.getSpeedGradeMobile() >= ACHIEVEMENT_MAX_STAR
                && (!achievementsOld.hasSpeedGradeMobile() || achievementsOld.getSpeedGradeMobile() < ACHIEVEMENT_MAX_STAR))) {
            achievements.add(new DigestData.Achievement(tld, AchievementType.MOBILE_SPEED));
        }

        if (!achievementsOld.getHttps() && achievementsNew.getHttps()) {
            achievements.add(new DigestData.Achievement(tld, AchievementType.HTTPS));
        }

        if (achievementsNew.hasPopular() && achievementsNew.getPopular() >= ACHIEVEMENT_MAX_STAR
                && (!achievementsOld.hasPopular() || achievementsOld.getPopular() < ACHIEVEMENT_MAX_STAR)) {
            achievements.add(new DigestData.Achievement(tld, AchievementType.POPULAR));
        }

        String oldName = achievementsOld.hasServiceCenter() ? achievementsOld.getServiceCenter() : null;
        String newName = achievementsNew.hasServiceCenter() ? achievementsNew.getServiceCenter() : null;
        if (!StringUtils.equals(oldName, newName) && !StringUtils.isEmpty(newName)) {
            achievements.add(new DigestData.BrandedOfficialAchievement(tld, AchievementOfficialType.SERVICE_CENTER, newName));
        }

        if (!achievementsOld.getTasix() && achievementsNew.getTasix()) {//TODO
            achievements.add(new DigestData.Achievement(tld, AchievementType.TAS_IX));
        }

        if (achievementsNew.hasUserSelection() && achievementsNew.getUserSelection() >= ACHIEVEMENT_MAX_STAR
                && (!achievementsOld.hasUserSelection() || achievementsOld.getUserSelection() < ACHIEVEMENT_MAX_STAR)) {
            achievements.add(new DigestData.Achievement(tld, AchievementType.USER_CHOICE));
        }

        // TODO Https,Turbo, SpeedDesktop, AbsoluteSpeedDesktop, SpeedMobile, AbsoluteSpeedMobile
        return Pair.of(tld, achievements);
    }

    private String urlWithoutSchema(String url) {
        int schemaEnd = url.indexOf(':');
        if (schemaEnd == -1) {
            return "";
        }
        return url.substring(schemaEnd + 1);
    }

    private boolean regionsEquals(List<Digest.GeoRegionInfo> regionsListOld, List<Digest.GeoRegionInfo> regionsListNew) {
        List<Long> oldIds = regionsListOld.stream().map(Digest.GeoRegionInfo::getRegionId).sorted().collect(Collectors.toList());
        List<Long> newIds = regionsListNew.stream().map(Digest.GeoRegionInfo::getRegionId).sorted().collect(Collectors.toList());
        return oldIds.equals(newIds);
    }

    public void withSearchable(long prevCount, long count, long added, long removed, List<Digest.SitetreeUrlHistory> urlHistoryList) {
        List<DigestData.Searchable.Sample> samples = createSearchableSamples(urlHistoryList);
        DigestData.Searchable searchableLocal = samples.isEmpty() ? null :
                new DigestData.Searchable(prevCount, count, new DigestData.Searchable.Stats(added, removed), samples);
        this.searchable = Optional.ofNullable(searchableLocal);
    }

    public void withAchievements(List<Digest.AchievementsSnapshot> achievementsOld, List<Digest.AchievementsSnapshot> achievementsNew) {
        List<Pair<AchievementTld, List<DigestData.AbstractAchievement>>> achievements = createAchievements(achievementsOld, achievementsNew);

        this.achievements = Optional.ofNullable(achievements);
    }

    private List<DigestData.Searchable.Sample> createSearchableSamples(List<Digest.SitetreeUrlHistory> urlHistory) {
        List<DigestData.Searchable.Sample> samples = new ArrayList<>();
        List<Digest.SitetreeUrlHistory> urlHistoryList = new ArrayList<>();
        Set<String> paths = new HashSet<>();
        for (int i = urlHistory.size() - 1; i >= 0; i--) {
            Digest.SitetreeUrlHistory sitetreeUrlHistory = urlHistory.get(i);
            if (!paths.contains(sitetreeUrlHistory.getPath())) {
                paths.add(sitetreeUrlHistory.getPath());
                urlHistoryList.add(sitetreeUrlHistory);
            }
        }
        Collections.reverse(urlHistoryList);
        for (Digest.SitetreeUrlHistory history : urlHistoryList.subList(Math.max(urlHistoryList.size() - 3, 0), urlHistoryList.size())) {
            String excludedReasonString = history.getIsAdded() ? "" : i18n.getSearchableReason(SearchUrlStatusUtil.viewFromJupiterStatusId((int) history.getUrlStatus(), history.getIsAdded()));
            samples.add(new DigestData.Searchable.Sample(history.getTitle(), history.getPath(),
                    history.getIsAdded() ? DigestData.StatsEnum.ADDED : DigestData.StatsEnum.REMOVED,
                    excludedReasonString));
        }
        return samples;
    }

    public void withClicks(long oldClicksByPeriodCount, long newClicksByPeriodCount, boolean oldDataPresent) {
        Long oldClicks = oldDataPresent ? oldClicksByPeriodCount : null;
        DigestData.Clicks clicks = newClicksByPeriodCount == 0 ? null :
                new DigestData.Clicks(newClicksByPeriodCount, Optional.ofNullable(oldClicks));
        this.clicks = Optional.ofNullable(clicks);
    }

    public void withQueries(List<Digest.QueryClicksSample> negativeQueryClicksSamples, List<Digest.QueryClicksSample> positiveQueryClicksSamples) {
        List<Digest.QueryClicksSample> allSamples = new ArrayList<>();
        allSamples.addAll(negativeQueryClicksSamples);
        allSamples.addAll(positiveQueryClicksSamples);
        allSamples.sort(Comparator.comparing(sample -> -Math.abs(sample.getClicks().getNewValue() - sample.getClicks().getOldValue())));

        this.queries = Optional.ofNullable(allSamples.isEmpty() ? null : new DigestData.Queries(
                allSamples.stream()
                        .limit(5)
                        .map(sample -> new DigestData.Queries.QData(
                                sample.getQuery(),
                                sample.getClicks().getNewValue(),
                                sample.getClicks().getNewValue() - sample.getClicks().getOldValue())).collect(Collectors.toList())
        ));
    }

    public void withQueriesGroups(List<Digest.GroupClicksSample> negativeGroupClicksSamples, List<Digest.GroupClicksSample> positiveGroupClicksSamples) {
        List<Digest.GroupClicksSample> allSamples = new ArrayList<>();
        allSamples.addAll(negativeGroupClicksSamples);
        allSamples.addAll(positiveGroupClicksSamples);
        allSamples.sort(Comparator.comparing(sample -> -Math.abs(sample.getClicks().getNewValue() - sample.getClicks().getOldValue())));

        List<DigestData.QueriesGroups.QDataGroups> groups = allSamples.stream()
                .limit(5)
                .flatMap(sample -> {
                    SpecialGroup specialGroup = getSpecialGroup(sample);
                    String groupName = specialGroup == SpecialGroup.UNKNOWN ?
                            sample.getGroupName() : i18n.getSpecialGroupName(specialGroup);
                    if (specialGroup == SpecialGroup.ALL_QUERIES || specialGroup == SpecialGroup.UPLOADED_QUERIES) {
                        return Stream.empty();
                    }
                    return Stream.of(new DigestData.QueriesGroups.QDataGroups(
                            groupName,
                            sample.getClicks().getNewValue(),
                            sample.getClicks().getNewValue() - sample.getClicks().getOldValue()));
                }).collect(Collectors.toList());
        this.queriesGroups = Optional.ofNullable(groups.isEmpty() ? null : new DigestData.QueriesGroups(
                groups
        ));
    }

    private static SpecialGroup getSpecialGroup(Digest.GroupClicksSample groupInfo) {
        UUID groupId = UUID.fromString(groupInfo.getGroupId());
        return SpecialGroup.byUUID(groupId);
    }

    public void withImportantUrl(List<Digest.ImportantUrlEvent> eventsList) {
        //TODO test
        List<DigestData.ImportantUrl> importantUrls =
                eventsList.stream()
                        .collect(Collectors.toMap(
                                Digest.ImportantUrlEvent::getPath,
                                Function.identity(),
                                (a, b) -> a,
                                LinkedHashMap::new // убираем дубли урлов, сохраняя порядок
                        )).values()
                        .stream()
                        .limit(3)
                        .map(event -> {
                            Digest.ImportantUrlState oldState = event.getOldState();
                            Digest.ImportantUrlState newState = event.getNewState();
                            SearchUrlStatusEnum oldStatus = SearchUrlStatusUtil.viewFromJupiterStatusId((int) oldState.getUrlStatus(), oldState.getIsSearchable());
                            SearchUrlStatusEnum newStatus = SearchUrlStatusUtil.viewFromJupiterStatusId((int) newState.getUrlStatus(), newState.getIsSearchable());
                            return new DigestData.ImportantUrl(
                                    event.getPath(),
                                    newState.getTitle(),
                                    new DigestData.ImportantUrl.HighlightedValue(
                                            Long.toString(newState.getHttpCode()),
                                            newState.getHttpCode() != oldState.getHttpCode()),
                                    new DigestData.ImportantUrl.HighlightedValue(
                                            i18n.getSearchableReason(newStatus),
                                            newStatus != oldStatus));
                        })
                        .collect(Collectors.toList());
        DigestData.ImportantUrls importantUrl = importantUrls.isEmpty() ? null : new DigestData.ImportantUrls(importantUrls);
        this.importantUrl = Optional.ofNullable(importantUrl);
    }

    public void withRecommendedUrl(List<Digest.RecommendedUrlState> urlList) {
        List<DigestData.RecommendedUrl> recommendedUrlsList = urlList.stream()
                .sorted(Comparator.comparing(Digest.RecommendedUrlState::getClicks, Comparator.reverseOrder())
                        .thenComparing(Digest.RecommendedUrlState::getShows, Comparator.reverseOrder()))
                .map(url -> new DigestData.RecommendedUrl(
                        url.getPath(),
                        url.getShows(),
                        url.getClicks()))
                .collect(Collectors.toList());
        DigestData.RecommendedUrls recommendedUrls = new DigestData.RecommendedUrls(recommendedUrlsList);
        this.recommendedUrl = Optional.of(recommendedUrls);
    }

    public void withBlogPosts(List<DigestData.BlogPost> blogPostList) {
        DigestData.BlogPosts blogPosts = (blogPostList.isEmpty() || i18n.language != LanguageEnum.RU) ? null
                : new DigestData.BlogPosts(blogPostList);
        this.blogPosts = Optional.ofNullable(blogPosts);
    }

    public void withIsFirstLiteDigest(boolean isFirstLiteDigest) {
        this.isFirstLiteDigest = Optional.ofNullable(isFirstLiteDigest);
    }

    public void withReviews(Digest.Reviews reviews){
        if (reviews == null) {
            this.reviews = Optional.empty();
            return;
        }

        long oldReviews = reviews.getCountOld();
        long newReviews = reviews.getCountNew();
        DigestData.Reviews deltaReviews = null;

        if (oldReviews < newReviews){
            deltaReviews = new DigestData.Reviews(newReviews - oldReviews, newReviews);
        }

        this.reviews = Optional.ofNullable(deltaReviews);
    }

    public void withMirrors(Digest.Mirrors mirrors){
        if (mirrors == null) {
            this.mirrors = Optional.empty();
            return;
        }

        String mainMirror = mirrors.getMainMirror();

        if (!StringUtils.isEmpty(mainMirror)){
            this.mirrors = Optional.of(new DigestData.Mirrors(mainMirror));
        } else {
            this.mirrors = Optional.empty();
        }
    }

    public void withMetrikaProblem(List<Integer> currentProblems) {
        Optional<DigestData.MetrikaCounterProblem> opt = currentProblems.stream()
                .map(SiteProblemTypeEnum.R::fromValueOrNull)
                .filter(Objects::nonNull)
                .filter(problemType -> problemType == SiteProblemTypeEnum.NO_METRIKA_COUNTER_BINDING ||
                        problemType == SiteProblemTypeEnum.NO_METRIKA_COUNTER_CRAWL_ENABLED)
                .findAny()
                .map(DigestData.MetrikaCounterProblem::new);

        this.metrikaCounterProblem = opt;
    }

    public DigestData build() {
        return new DigestData(host, dateFrom, dateTo,
                checklist,
                searchable,
                clicks,
                queries,
                queriesGroups,
                importantUrl,
                recommendedUrl,
                blogPosts,
                achievements,
                reviews,
                mirrors,
                metrikaCounterProblem,
                isFirstLiteDigest
        );
    }

}
