package ru.yandex.webmaster3.worker.digest.html;

import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.tanker.TankerCommonTexts;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.function.IntFunction;

/**
 * @author avhaliullin
 */
public class DigestTextUtil {
    private static final DecimalFormat DECIMAL_FORMAT = new DecimalFormat();
    private static final String LONG_DASH = "—";
    private static final String SHORT_DASH = "–";

    static {
        DECIMAL_FORMAT.setGroupingSize(3);
        DECIMAL_FORMAT.setGroupingUsed(true);
        DecimalFormatSymbols dfs = DECIMAL_FORMAT.getDecimalFormatSymbols();
        dfs.setGroupingSeparator(' ');
        DECIMAL_FORMAT.setDecimalFormatSymbols(dfs);
    }

    /**
     * Разделяет число по разрядам пробелами.
     *
     * @param {Long} value Любой текст содержащий числа
     * @example 10000 -> 10 000
     * @returns {String}
     */
    public static String formatNumber(long value) {
        return DECIMAL_FORMAT.format(value);
    }

    public static String readableHost(WebmasterHostId hostId) {
        return IdUtils.toHostString(hostId, false, true, false);
    }

    public static String shortDateRange(LanguageEnum lang, LocalDate fromDate, LocalDate toDate) {
        return writeRange(fromDate, toDate, month -> TankerCommonTexts.monthShortGenitive(lang, month));
    }

    public static String fullDateRange(LanguageEnum lang, LocalDate fromDate, LocalDate toDate) {
        return writeRange(fromDate, toDate, month -> TankerCommonTexts.monthGenitiveLowerCase(lang, month));
    }

    private static String writeRange(LocalDate fromDate, LocalDate toDate, IntFunction<String> monthConverter) {
        if (fromDate.getMonthOfYear() == toDate.getMonthOfYear()) {
            return String.format("%d " + SHORT_DASH + " %d %s", fromDate.getDayOfMonth(), toDate.getDayOfMonth(), monthConverter.apply(toDate.getMonthOfYear()));
        } else {
            return String.format("%d %s " + LONG_DASH + " %d %s",
                    fromDate.getDayOfMonth(), monthConverter.apply(fromDate.getMonthOfYear()),
                    toDate.getDayOfMonth(), monthConverter.apply(toDate.getMonthOfYear())
            );
        }
    }

    public static String dateRangeInSubject(LanguageEnum lang, LocalDate fromDate, LocalDate toDate) {
        return fullDateRange(lang, fromDate, toDate);
    }

    public static String dateRangeInHeader(LanguageEnum lang, LocalDate fromDate, LocalDate toDate) {
        return fullDateRange(lang, fromDate, toDate);
    }

    public static String dateInPost(LanguageEnum lang, DateTime postDate) {
        String month = TankerCommonTexts.monthGenitiveLowerCase(lang, postDate.getMonthOfYear());
        DateTimeFormatter timeFormatter = DateTimeFormat.forPattern("HH:mm");
        return String.format("%d %s, %s", postDate.getDayOfMonth(), month, timeFormatter.print(postDate));
    }
}
