package ru.yandex.webmaster3.worker.download;

import java.io.IOException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.download.DownloadInfo;
import ru.yandex.webmaster3.storage.download.DownloadInfoYDao;
import ru.yandex.webmaster3.storage.download.DownloadTaskData;
import ru.yandex.webmaster3.storage.download.MDSService;

/**
 * Created by ifilippov5 on 15.02.17.
 */
public class DownloadRunnerService {
    private static final Logger log = LoggerFactory.getLogger(DownloadRunnerService.class);
    private DownloadInfoYDao downloadInfoYDao;
    private MDSService mdsService;

    public <T> void run(DownloadTaskData<T> data, List<T> samples) {
        Long hash = data.getHash();
        String uri;
        try {
            uri = download(data, samples);
        } catch (Exception e) {
            log.error("Error uploading samples", e);
            log.warn("Download task with hostId = {} and hash = {} terminate in status Internal Error",
                    data.getHostId(), hash);
            downloadInfoYDao.add(data.getHostId(), hash, DownloadInfo.error());
            throw e;
        }
        log.debug("Download task with hostId = {} and hash = {} complete successfully", data.getHostId(), hash);
        downloadInfoYDao.add(data.getHostId(), hash, DownloadInfo.success(data.getFileName(), uri));
    }

    private <T> String download(DownloadTaskData<T> data, List<T> samples) {
        log.debug("Number of rows: {}", samples.size());
        byte[] output = null;
        try {
            log.debug("Save export");
            output = data.getExportFormat().export(samples, data.getType());
            String url = mdsService.uploadFileAndGetDownloadLink(output,
                    data.getFileName() + data.getExportFormat().getExtension());
            log.debug("Export saved to: {}", url);
            return url;
        } catch (IOException e) {
            log.error("Unable to upload to MDS", e);
            throw new WebmasterException("Unable export data to file",
                    new WebmasterErrorResponse.ExportErrorResponse(getClass(), e), e);
        }
    }

    @Required
    public void setDownloadInfoYDao(DownloadInfoYDao downloadInfoYDao) {
        this.downloadInfoYDao = downloadInfoYDao;
    }

    @Required
    public void setMdsService(MDSService mdsService) {
        this.mdsService = mdsService;
    }
}
