package ru.yandex.webmaster3.worker.favicon;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Range;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.environment.YandexEnvironmentProvider;
import ru.yandex.webmaster3.core.util.environment.YandexEnvironmentType;
import ru.yandex.webmaster3.proto.Favicons;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsNotificationService;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.favicon.HostFaviconsCHDao;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.turbo.AbstractYqlPrepareImportTask;

import static ru.yandex.webmaster3.storage.host.CommonDataType.LAST_IMPORTED_HOST_FAVICONS_CHANGES;

/**
 * Created by Oleg Bazdyrev on 03/08/2020.
 */
@Slf4j
public class ImportHostFaviconsTask extends AbstractYqlPrepareImportTask {

    private static final int LINES_COUNT = 128;
    private static final Duration PROCESSED_MAX_DATA_AGE = Duration.standardDays(5L);

    @Autowired
    private SettingsService settingsService;
    @Autowired
    private SiteProblemsNotificationService siteProblemsNotificationService;
    @Autowired
    private WMCEventsService wmcEventsService;


    @Value("${webmaster3.worker.favicons.import.changesDir}")
    private YtPath changesDir;

    @Override
    protected CHTable getTable() {
        return HostFaviconsCHDao.TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.HOST_FAVICONS;
    }

    @Override
    public int getShardsCount() {
        return 1;
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        int shardCount = getShardsCount();
        String dateString = IN_YQL_QUERY_DATE_FORMATTER.print(imprt.getDateTo());

        return YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6)" +
                        " as data FROM (\n")
                .appendText("SELECT")
                .appendText("(Digest::Fnv64(").appendFCall(YqlFunctions.url2HostId("Host")).appendText(") % " + shardCount + ") as ShardId,")
                .appendText("((Digest::Fnv64(").appendFCall(YqlFunctions.url2HostId("Host")).appendText(") / " + shardCount + ") % " + LINES_COUNT + ") as RowId,")
                .appendText("('" + dateString + "' || '\\t' || ")
                .appendFCall(YqlFunctions.url2HostId("Host"))
                .appendText(" || '\\t' || String::Base64Encode(FaviconsAndProblems) || '\\n') as data ")
                .appendText("FROM")
                .appendTable(tablePath)
                .appendText("view raw) \n GROUP BY ShardId, RowId;")
                .appendText("COMMIT;\n\n");
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        // импорт изменений и рассылка оповещений
        // у нас есть защита от повторных отправок, так что не будем заморачиваться с ретраями

        ytService.withoutTransaction(cypressService -> {
            long lastProcessedTimestamp =
                    Optional.ofNullable(settingsService.getSettingUncached(LAST_IMPORTED_HOST_FAVICONS_CHANGES))
                            .map(CommonDataState::getValue).map(Long::valueOf).orElse(0L);
            List<YtPath> tablesToProcess = cypressService.list(changesDir);
            tablesToProcess.sort(null);
            log.info("Found {} changes tables to process: {}", tablesToProcess.size(), tablesToProcess);
            for (YtPath table : tablesToProcess) {
                long tableTimestamp = Long.parseLong(table.getName());
                if (tableTimestamp <= lastProcessedTimestamp) {
                    continue;
                }
                // process notifications
                // TODO turn on after check
                processChanges(cypressService, table);

                lastProcessedTimestamp = tableTimestamp;
                settingsService.update(LAST_IMPORTED_HOST_FAVICONS_CHANGES, String.valueOf(lastProcessedTimestamp));
            }

            if (YandexEnvironmentProvider.getEnvironmentType() == YandexEnvironmentType.PRODUCTION) {
                cleanup(cypressService);
            }

            return true;
        });

        return super.rename(imprt);
    }

    private void processChanges(YtCypressService cypressService, YtPath table) throws InterruptedException {
        log.info("Processing changes table {}", table);
        AsyncTableReader<ChangedRecord> tableReader = new AsyncTableReader<>(cypressService, table, Range.all(),
                YtTableReadDriver.createYSONDriver(ChangedRecord.class)).withThreadName("favicons-changes-reader");
        try (var iterator = tableReader.read()) {
            while (iterator.hasNext()) {
                ChangedRecord row = iterator.next();
                WebmasterHostId hostId = IdUtils.urlToHostId(row.getHost());
                // отсылаем сообщения, если надо
                var faviconsAndProblems = Favicons.THostFaviconsAndProblems.parseFrom(row.getData());
                List<WMCEventContent> events = new ArrayList<>();
                for (var problem : faviconsAndProblems.getProblemsList()) {
                    DateTime lastUpdate = problem.getLastUpdate() == 0L ? null : new DateTime(problem.getLastUpdate());
                    DateTime actualSince = problem.getActualSince() == 0L ? null : new DateTime(problem.getActualSince());
                    SiteProblemContent content = null;
                    switch (problem.getType()) {
                        case MISSING_FAVICON:
                            content = new SiteProblemContent.MissingFavicon();
                            break;
                        case FAVICON_ERROR:
                            content = new SiteProblemContent.FaviconError();
                            break;
                        case BIG_FAVICON_ABSENT:
                            content = new SiteProblemContent.BigFaviconAbsent();
                            break;
                    }
                    if (content != null) {
                        events.add(siteProblemsNotificationService.createUserMessageEvent(hostId, new ProblemSignal(content, lastUpdate), null, actualSince));
                    }
                }
                events = events.stream().filter(Objects::nonNull).collect(Collectors.toList());
                if (!events.isEmpty()) {
                    log.info("Sending favicon notifications for host {}", row.getHost());
                    wmcEventsService.addEventContents(events);
                }
            }
        } catch (IOException e) {
            throw new WebmasterException("Error reading table",
                    new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }
    }

    private void cleanup(YtCypressService cypressService) {
        long maxTimestamp = DateTime.now().minus(PROCESSED_MAX_DATA_AGE).getMillis();
        for (YtPath table : cypressService.list(changesDir)) {
            long tableTimestamp = Long.parseLong(table.getName());
            if (tableTimestamp < maxTimestamp) {
                log.info("Removing old table {}", table);
                cypressService.remove(table);
            }
        }
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.HOST_FAVICONS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 38 * * * *");
    }

    @lombok.Value
    private static final class ChangedRecord {
        @JsonProperty("Host")
        String host;
        @JsonProperty("FaviconsAndProblems")
        byte[] data;
    }
}
