package ru.yandex.webmaster3.worker.feedback;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.feedback.Feedback;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.feedback.dao.FeedbackYDao;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.host.CommonDataType.FEEDBACK_LAST_EXPORT_DATE;

/**
 * @author: ishalaru
 * DATE: 03.06.2019
 */
public class ExportFeedbackTask extends PeriodicTask<ExportFeedbackTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(ExportFeedbackTask.class);
    private static final TaskSchedule TASK_SCHEDULE = TaskSchedule.startByCron("0 0 2 * * *");

    private final CommonDataStateYDao commonDataStateYDao;
    private final FeedbackYDao feedbackYDao;
    private final YtService ytService;
    private YtPath tablePath;

    @Autowired
    public ExportFeedbackTask(CommonDataStateYDao commonDataStateYDao,
                              FeedbackYDao feedbackYDao,
                              YtService ytService) {
        this.commonDataStateYDao = commonDataStateYDao;
        this.feedbackYDao = feedbackYDao;
        this.ytService = ytService;
    }

    public void setTablePath(YtPath tablePath) {
        this.tablePath = tablePath;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Exporting feedbacks to YT table");

        TaskState state = new TaskState();
        setState(state);

        DateTime lastRunTime =
                Optional.ofNullable(commonDataStateYDao.getValue(CommonDataType.FEEDBACK_LAST_EXPORT_DATE))
                        .map(CommonDataState::getValue).map(Long::parseLong).map(DateTime::new)
                        .orElse(new DateTime(Long.MIN_VALUE));
        DateTime taskStart = DateTime.now();

        YtTableData tableData = ytService.prepareTableData("feedbacks", tableWriter -> {
                    try {
                        feedbackYDao.forEach(e -> {
                            writeBatch(e, tableWriter, lastRunTime, taskStart);
                        });
                    } catch (WebmasterYdbException e) {
                        throw new WebmasterException("Ydb error",
                                new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
                    } catch (YtException e) {
                        throw new WebmasterException("YT error",
                                new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                    }
                }
        );

        ytService
                .inTransaction(tablePath)
                .execute(cypressService -> uploadToYt(cypressService, tablePath, tableData));

        // чистим за собой
        tableData.delete();
        commonDataStateYDao.update(new CommonDataState(
                FEEDBACK_LAST_EXPORT_DATE,
                String.valueOf(taskStart.getMillis()),
                DateTime.now()));

        // отправляем статистику в Solomon
        return new Result(TaskResult.SUCCESS);
    }


    private boolean uploadToYt(YtCypressService cypressService, YtPath path,
                               YtTableData tableData)
            throws YtException {
        if (!cypressService.exists(path)) {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
            cypressService.create(path, YtNode.NodeType.TABLE, true, attributes, true);
        }
        cypressService.writeTable(path, tableData, true);
        return true;
    }


    private void writeBatch(Pair<String, Feedback> pair, TableWriter tw, DateTime startValidDate,
                            DateTime endValidDate) throws YtException {
        Feedback feedback = pair.getRight();
        if (feedback.getCreateDate().isAfter(startValidDate) && feedback.getCreateDate().isBefore(endValidDate)) {
            String domain = pair.getLeft();
            F.HOST_ID.set(tw, domain);
            F.USER_ID.set(tw, feedback.getUserId());
            F.CREATE_DATE.set(tw, feedback.getCreateDate().getMillis());
            F.FEEDBACK_TYPE.set(tw, feedback.getType().toString());
            F.MESSAGE.set(tw, feedback.getMessage());
            F.META.set(tw, feedback.getMeta());
            tw.rowEnd();
            state.newFeedbacksCount++;
        } else {
            state.outOfDateFeedbacksCount++;
        }
        state.totalFeedbacksCount++;

    }


    public static class TaskState implements PeriodicTaskState {

        private long totalFeedbacksCount;
        private long outOfDateFeedbacksCount;
        private long newFeedbacksCount;
        private List<YtPath> tablesSuccess = new ArrayList<>();
        private List<YtPath> tablesFailed = new ArrayList<>();

        public long getTotalFeedbacksCount() {
            return totalFeedbacksCount;
        }

        public long getOutOfDateFeedbacksCount() {
            return outOfDateFeedbacksCount;
        }

        public long getNewFeedbacksCount() {
            return newFeedbacksCount;
        }

        public List<YtPath> getTablesSuccess() {
            return tablesSuccess;
        }

        public List<YtPath> getTablesFailed() {
            return tablesFailed;
        }
    }

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> HOST_ID = SCHEMA.addColumn("Host", YtColumn.Type.STRING);
        YtColumn<Long> USER_ID = SCHEMA.addColumn("userId", YtColumn.Type.INT_64);
        YtColumn<Long> CREATE_DATE = SCHEMA.addColumn("create_date", YtColumn.Type.INT_64);
        YtColumn<String> FEEDBACK_TYPE = SCHEMA.addColumn("type", YtColumn.Type.STRING);
        YtColumn<String> MESSAGE = SCHEMA.addColumn("message", YtColumn.Type.STRING);
        YtColumn<Object> META = SCHEMA.addColumn("meta", YtColumn.Type.any());
    }

    private static DateTime extractDateFromTableName(String name) {
        // Таблица называется по схеме "<первый день, включительно>_<последний день, не включительно>"
        return DateTime.parse(name.split("_")[1], ISODateTimeFormat.basicDate());
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_FEEDBACK;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TASK_SCHEDULE;
    }

}
