package ru.yandex.webmaster3.worker.feeds;

import java.util.UUID;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.feeds.feed_domain.FeedDomainInfo;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.feeds.FeedsDomainInfoYDao;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;


/**
 * @author kravchenko99
 * @date 8/20/21
 */


@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class ExportFeedsDomainsInfoTask extends PeriodicTask<ExportFeedsDomainsInfoTask.TaskState> {

    private final FeedsDomainInfoYDao feedsDomainInfoYDao;
    private final YtService ytService;
    @Value("${external.yt.service.hahn.root.default}/export/feeds/feeds_domains_info")
    private YtPath ytPath;

    @Override
    public Result run(UUID runId) throws Exception {

        YtTableData tableData = ytService.prepareTableData("feeds_domains_info", tableWriter -> {
            feedsDomainInfoYDao.forEach(info -> write(info, tableWriter));
        });
        try {
            ytService.inTransaction(ytPath).execute(cypressService -> uploadToYt(cypressService, ytPath, tableData));
        } finally {
            tableData.delete();
        }
        return new Result(TaskResult.SUCCESS);
    }


    private static boolean uploadToYt(YtCypressService cypressService, YtPath path,
                                      YtTableData tableData)
            throws YtException {
        YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
        cypressService.create(path, YtNode.NodeType.TABLE, true, attributes, true);
        cypressService.writeTable(path, tableData, false);
        return true;
    }

    private static void write(FeedDomainInfo info, TableWriter tw) throws YtException {
        F.DOMAIN.set(tw, info.getDomain());
        F.BUSINESS_ID.set(tw, info.getBusinessId());
        tw.rowEnd();
    }

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> DOMAIN = SCHEMA.addColumn("domain", YtColumn.Type.STRING);
        YtColumn<Long> BUSINESS_ID = SCHEMA.addColumn("business_id", YtColumn.Type.INT_64);
    }


    @Getter
    public static class TaskState implements PeriodicTaskState {

    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.FEEDS_EXPORT_DOMAINS_INFO_FEEDS;
    }

    @Override
    public TaskSchedule getSchedule() {
        //0 */5 * * * *
        return TaskSchedule.startByCron("0 21 * * * *");
    }
}
