package ru.yandex.webmaster3.worker.feeds;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableObject;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedsDefectRateErrorInfo;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.concurrent.graph.BlockingBatchConsumer;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphExecution;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphExecutionBuilder;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphOutQueue;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserDomainMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.feeds.FeedsDefectRateErrorYDao;
import ru.yandex.webmaster3.storage.feeds.FeedsDomainNotificationsYDao;
import ru.yandex.webmaster3.storage.feeds.FeedsDomainNotificationsYDao.FeedsDomainNotifications;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.models.FeedStats;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 19.07.2022.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UpdateFeedsProblemsTask extends PeriodicTask<PeriodicTaskState> {

    // TODO
    //private static int[] DEFECT_RATE_INTERVALS = {0, 3, 4};
    private static int[] SCC_INTERVALS = {0, 3, 4};
    private static int[] VALIDATION_INTERVALS = {0, 3, 4};

    private final FeedsDefectRateErrorYDao defectRateErrorYDao;
    private final FeedsDomainNotificationsYDao feedsDomainNotificationsYDao;
    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final FeedsService feedsService;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;
    private final WMCEventsService wmcEventsService;
    private final YtService ytService;

    private static long daysFrom(DateTime dt) {
        if (dt == null) {
            return Integer.MAX_VALUE;
        }
        return (System.currentTimeMillis() - dt.getMillis()) / 86_400_000;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        GraphExecutionBuilder builder = GraphExecutionBuilder.newBuilder("update-feeds-problems");
        GraphExecutionBuilder.Queue<FeedsDomainNotifications> stateUpdater = builder
                .process(() -> feedsDomainNotificationsYDao::update)
                .name("update-state")
                .batchLimit(1000)
                .getInput();
        GraphExecutionBuilder.Queue<Pair<FeedsDomainNotifications, FeedsDomainNotifications>> stateComparator = builder
                .process(stateUpdater, this::compareStateAndSentNotifications)
                .name("compare-and-sent")
                .batchLimit(1000)
                .getInput();

        GraphExecutionBuilder.Queue<FeedsDomainNotifications> previousStateJoiner = builder
                .process(stateComparator, this::findPreviousDomainFeedStatus)
                .name("previous-state-joiner")
                .batchLimit(100)
                .getInput();

        GraphExecutionBuilder.Queue<List<NativeFeedInfo2>> currentStateCalc = builder
                .process(previousStateJoiner, this::calcCurrentDomainFeedStatus)
                .name("current-state-calc")
                .batchLimit(100)
                .getInput();

        try (GraphExecution<List<NativeFeedInfo2>> graph = builder.build(currentStateCalc)) {
            graph.start();
            MutableObject<String> lastDomain = new MutableObject<>();
            List<NativeFeedInfo2> infos = new ArrayList<>();
            feedsNative2YDao.forEach(feedInfo -> {
                if (!Objects.equals(feedInfo.getDomain(), lastDomain.getValue())) {
                    if (!infos.isEmpty()) {
                        try {
                            graph.put(new ArrayList<>(infos));
                        } catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                            throw new RuntimeException(e);
                        }
                    }
                    lastDomain.setValue(feedInfo.getDomain());
                    infos.clear();
                }
                infos.add(feedInfo);
            });
            graph.doneWritingAndAwaitTermination();
        }
        return new Result(TaskResult.SUCCESS);
    }

    private BlockingBatchConsumer<Pair<FeedsDomainNotifications, FeedsDomainNotifications>> compareStateAndSentNotifications(
            GraphOutQueue<FeedsDomainNotifications> q) {
        return batch -> {
            for (var pair : batch) {
                FeedsDomainNotifications previous = Objects.requireNonNullElse(pair.getLeft(), FeedsDomainNotifications.EMPTY);
                FeedsDomainNotifications current = pair.getRight();
                WebmasterHostId hostId = IdUtils.urlToHostId(current.getDomain());
                var currentBuilder = previous.toBuilder()
                        .domain(current.getDomain()) // domain may be missing
                        .defectRateStatus(current.getDefectRateStatus())
                        .sccStatus(current.getSccStatus())
                        .validationStatus(current.getValidationStatus())
                        .lastUpdate(DateTime.now())
                        .validationFailedFeeds(current.getValidationFailedFeeds())
                        .sccFailedFeeds(current.getSccFailedFeeds());
                if (current.getDefectRateStatus().isBad() && !previous.getDefectRateStatus().isBad()) {
                    // на DEFECT_RATE нет ограничений на повторные оповещения
                    log.info("About to send defect rate notification for domain {}, severity = {}", current.getDomain(), current.getDefectRateStatus());
                    currentBuilder.defectRateNotifications(previous.getDefectRateNotifications() + 1);
                    currentBuilder.lastDefectRateNotification(DateTime.now());
                    sendMessage(new MessageContent.FeedsDefectRateFailed(hostId, current.getDefectRateStatus()));
                }
                if (current.getSccStatus() == FeedsErrorSeverity.FATAL) {
                    if (previous.getSccNotifications() < SCC_INTERVALS.length) {
                        if (daysFrom(previous.getLastSccNotification()) >= SCC_INTERVALS[previous.getSccNotifications()]) {
                            log.info("About to send scc notification for domain {}", current.getDomain());
                            currentBuilder.sccNotifications(previous.getSccNotifications() + 1);
                            currentBuilder.lastSccNotification(DateTime.now());
                            // first in PreModerationCheckResultsProcessing
                            if (previous.getSccNotifications() > 0) {
                                sendMessage(new MessageContent.FeedsSccFailed(hostId, current.getSccFailedFeeds(), previous.getSccNotifications()));
                            }
                        }
                    }
                } else if (current.getSccStatus() == FeedsErrorSeverity.SUCCESS) {
                    if (previous.getGoodSccNotifications() == 0) { // only first time
                        log.info("About to send good scc notification for domain {}", current.getDomain());
                        currentBuilder.goodSccNotifications(previous.getGoodSccNotifications() + 1);
                        sendMessage(new MessageContent.FeedsSccPassed(hostId, Collections.emptyList()));
                    }
                }

                if (current.getValidationStatus().isBad()) {
                    if (previous.getValidationNotifications() < VALIDATION_INTERVALS.length) {
                        if (daysFrom(previous.getLastValidationNotification()) >= VALIDATION_INTERVALS[previous.getValidationNotifications()]) {
                            log.info("About to send validation notification for domain {}", current.getDomain());
                            currentBuilder.validationNotifications(previous.getValidationNotifications() + 1);
                            currentBuilder.lastValidationNotification(DateTime.now());
                            sendMessage(new MessageContent.FeedsValidationFailed(hostId, current.getValidationFailedFeeds(),
                                    previous.getValidationNotifications()));
                        }
                    }
                }
                q.put(currentBuilder.build());

            }
        };
    }

    private BlockingBatchConsumer<List<NativeFeedInfo2>> calcCurrentDomainFeedStatus(GraphOutQueue<FeedsDomainNotifications> q) {
        return batch -> {
            for (List<NativeFeedInfo2> feedInfos : batch) {
                String domain = feedInfos.get(0).getDomain();
                List<String> sccFailedFeeds = new ArrayList<>();
                List<String> validationFailedFeeds = new ArrayList<>();
                // worst DR status
                FeedsErrorSeverity drStatus = defectRateErrorYDao.list(domain).stream().map(FeedsDefectRateErrorInfo::getSeverity)
                        .max(FeedsErrorSeverity.BY_ORDER).orElse(FeedsErrorSeverity.SUCCESS);
                // worst scc status
                FeedsErrorSeverity sccStatus = null;
                // worst validation status
                FeedsErrorSeverity validationStatus = FeedsErrorSeverity.SUCCESS;

                Map<String, FeedRecord> offersStateByUrl = feedsOffersLogsHistoryCHDao.getLastState(
                        feedInfos.stream().filter(NativeFeedInfo2::isNotGoodsFeed).map(NativeFeedInfo2::getUrl).toList()
                ).stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));
                Map<String, FeedRecord> serpdataStateByUrl = serpdataLogsHistoryCHDao.getLastState(domain, null, null)
                        .stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));
                var feedIds = feedInfos.stream().filter(NativeFeedInfo2::isGoodsFeed)
                        .filter(f -> f.getBusinessId() != null && f.getPartnerId() != null && f.getFeedId() != null)
                        .map(f -> Triple.of(f.getBusinessId(), f.getPartnerId(), f.getFeedId()))
                        .collect(Collectors.toList());
                goodsOffersLogsHistoryCHDao.getLastState(feedIds).forEach(goodsFeedRecord -> {
                    offersStateByUrl.put(goodsFeedRecord.getFeedUrl(), goodsFeedRecord.toFeedRecord());
                });

                for (NativeFeedInfo2 feedInfo : feedInfos) {
                    if (feedInfo.getStatusScc() == NativeFeedSccStatus.FAILED) {
                        sccFailedFeeds.add(feedInfo.getUrl());
                        sccStatus = FeedsErrorSeverity.FATAL;
                    } else if (feedInfo.getStatusScc() == NativeFeedSccStatus.SUCCESS && sccStatus == null) {
                        sccStatus = FeedsErrorSeverity.SUCCESS; // success only if all feeds is successfull
                    }
                    FeedRecord offerState = offersStateByUrl.get(feedInfo.getUrl());
                    FeedRecord serpdataState = serpdataStateByUrl.get(feedInfo.getUrl());
                    FeedsErrorSeverity severity = feedsService.getStatus(feedInfo.getType(), offerState, serpdataState).getSeverity();
                    if (severity.isBad()) {
                        validationFailedFeeds.add(feedInfo.getUrl());
                    }
                    if (severity.getOrder() > validationStatus.getOrder()) {
                        validationStatus = severity;
                    }
                }
                for (FeedRecord feedRecord : serpdataLogsHistoryCHDao.getLastState(domain, null)) {
                    FeedStats serpdataStats = Objects.requireNonNullElse(feedRecord.getErrorStats(), FeedStats.EMPTY);
                    if (serpdataStats.getError() > 0 || serpdataStats.getWarning() > 0) {
                        validationFailedFeeds.add("");
                        if (FeedsErrorSeverity.WARNING.getOrder() > validationStatus.getOrder()) {
                            validationStatus = FeedsErrorSeverity.WARNING;
                        }
                    }
                }
                // new status
                FeedsDomainNotifications current = FeedsDomainNotifications.builder()
                        .domain(domain)
                        .lastUpdate(DateTime.now())
                        .defectRateStatus(drStatus)
                        .sccStatus(sccStatus)
                        .validationStatus(validationStatus)
                        .validationFailedFeeds(validationFailedFeeds)
                        .sccFailedFeeds(sccFailedFeeds)
                        .build();
                q.put(current);
            }
        };
    }

    private BlockingBatchConsumer<FeedsDomainNotifications> findPreviousDomainFeedStatus(
            GraphOutQueue<Pair<FeedsDomainNotifications, FeedsDomainNotifications>> q) {
        return batch -> {
            List<String> domains = batch.stream().map(FeedsDomainNotifications::getDomain).collect(Collectors.toList());
            Map<String, FeedsDomainNotifications> map = feedsDomainNotificationsYDao.find(domains);
            for (var n : batch) {
                q.put(Pair.of(map.get(n.getDomain()), n));
            }
        };
    }

    public void sendMessage(MessageContent.HostMessageContent messageContent) {
        wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserDomainMessageEvent.create(
                messageContent.getHostId(),
                messageContent,
                NotificationType.FEEDS_INFO,
                false) // ?
        )));
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.FEEDS_UPDATE_PROBLEMS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }
}
