package ru.yandex.webmaster3.worker.feeds.download;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Strings;
import com.opencsv.CSVReader;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.feeds.feed.FeedsValidationErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedSccStatus;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.util.GzipUtils;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.util.functional.ThrowingConsumer;
import ru.yandex.webmaster3.storage.download.common.MdsExportTaskData;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.download.FeedsAllErrorsMdsExportDescriptor;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.feeds.logs.SerpdataLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableRange;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.download.AbstractMdsDataProvider;

import static ru.yandex.webmaster3.core.util.functional.ThrowingConsumer.rethrowingUnchecked;

/**
 * Created by Oleg Bazdyrev on 18/01/2022.
 */
@Service
@AllArgsConstructor(onConstructor_ = @Autowired)
public class FeedsAllErrorsDownloadMdsDataProvider extends AbstractMdsDataProvider<FeedsAllErrorsCsvRow> {

    private static final Pattern LEGACY_SEVERITY_LINE_PATTERN = Pattern.compile("([a-zA-Z]+)([0-9]*)");

    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final FeedsNative2YDao feedsNative2YDao;
    private final FeedsOffersLogsHistoryCHDao feedsOffersLogsHistoryCHDao;
    private final SerpdataLogsHistoryCHDao serpdataLogsHistoryCHDao;
    private final YtService ytService;
    @Value("hahn://home/webmaster/prod/feeds/offers/archive")
    private final YtPath feedsOffersArchive;
    @Value("hahn://home/webmaster/prod/feeds/serpdata/archive")
    private final YtPath feedsSerpdataArchive;

    @Override
    public void provide(MdsExportTaskData data, ThrowingConsumer<FeedsAllErrorsCsvRow, Exception> consumer) throws Exception {
        FeedsAllErrorsMdsExportDescriptor descriptor = (FeedsAllErrorsMdsExportDescriptor) data.getDescriptor();
        String domain = descriptor.getDomain();
        // для каждого фида добавляем ошибки СКК
        List<NativeFeedInfo2> feedsInfo = feedsNative2YDao.list(domain).stream().filter(feedInfo -> feedInfo.getType() == descriptor.getFeedType())
                .sorted(Comparator.comparing(NativeFeedInfo2::getUrl)).toList();
        List<String> feedUrls = feedsInfo.stream().map(NativeFeedInfo2::getUrl).collect(Collectors.toList());
        Map<String, FeedRecord> offersStateByUrl = feedsOffersLogsHistoryCHDao.getLastState(feedUrls)
                .stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));
        Map<String, FeedRecord> serpdataStateByUrl = serpdataLogsHistoryCHDao.getLastState(domain, null, null)
                .stream().filter(Objects::nonNull).collect(Collectors.toMap(FeedRecord::getUrl, Function.identity()));

        var feedIds = feedsInfo.stream().filter(f -> f.getType() == NativeFeedType.STORES)
                .filter(f -> f.getBusinessId() != null && f.getPartnerId() != null && f.getFeedId() != null)
                .map(f -> Triple.of(f.getBusinessId(), f.getPartnerId(), f.getFeedId()))
                .collect(Collectors.toList());
        goodsOffersLogsHistoryCHDao.getLastState(feedIds).forEach(goodsFeedRecord -> {
            offersStateByUrl.put(goodsFeedRecord.getFeedUrl(), goodsFeedRecord.toFeedRecord());
        });

        for (NativeFeedInfo2 feedInfo : feedsInfo) {
            if (feedInfo.getStatusScc() != NativeFeedSccStatus.SUCCESS) {
                for (String errorScc : feedInfo.getErrorsScc()) {
                    rethrowingUnchecked(consumer).accept(new FeedsAllErrorsCsvRow(feedInfo.getUrl(), "Проверка службы качества", "error",
                            null, null, null, errorScc, null));
                }
            }
        }
        ExecutorService executorService = Executors.newFixedThreadPool(4);
        List<Future<List<FeedsAllErrorsCsvRow>>> futures = new ArrayList<>();
        // ошибки валидации
        ytService.withoutTransaction((cypressService) -> {
            for (NativeFeedInfo2 feedInfo : feedsInfo) {
                var offerState = offersStateByUrl.get(feedInfo.getUrl());
                if (offerState != null) {
                    if (feedInfo.getType() == NativeFeedType.STORES) {
                        // get from CH
                        futures.add(executorService.submit(() -> convertErrors(offerState)));
                    } else {
                        AsyncTableReader<FeedsErrorsData> reader = new AsyncTableReader<>(
                                cypressService,
                                feedsOffersArchive,
                                YtTableRange.multiKey(WwwUtil.cutWWWAndM(IdUtils.urlToHostId(feedInfo.getUrl())), feedInfo.getUrl(), offerState.getTimestamp().getMillis() / 1000),
                                YtTableReadDriver.createYSONDriver(FeedsErrorsData.class)
                        ).withRetry(3).splitInParts(10000L);
                        futures.add(executorService.submit(() -> readFeedErrors(feedInfo, reader)));
                    }
                }

                var serpdataState = serpdataStateByUrl.get(feedInfo.getUrl());
                if (serpdataState != null) {
                    AsyncTableReader<FeedsErrorsData> reader = new AsyncTableReader<>(
                            cypressService,
                            feedsSerpdataArchive,
                            YtTableRange.multiKey(descriptor.getDomain(), feedInfo.getUrl(), serpdataState.getTimestamp().getMillis() / 1000),
                            YtTableReadDriver.createYSONDriver(FeedsErrorsData.class)
                    ).withRetry(3).splitInParts(10000L);

                    futures.add(executorService.submit(() -> readFeedErrors(feedInfo, reader)));
                }
            }
            return true;
        });
        // empty row at the end
        for (var future : futures) {
            future.get().forEach(rethrowingUnchecked(consumer));
        }
        rethrowingUnchecked(consumer).accept(new FeedsAllErrorsCsvRow(null, null, null, null, null, null, null, null));
    }

    private List<FeedsAllErrorsCsvRow> readFeedErrors(NativeFeedInfo2 feedInfo, AsyncTableReader<FeedsErrorsData> reader) throws InterruptedException {
        List<FeedsErrorsData> dataList = new ArrayList<>();
        List<FeedsAllErrorsCsvRow> result = new ArrayList<>();
        try (AsyncTableReader.TableIterator<FeedsErrorsData> read = reader.read()) {
            while (read.hasNext()) {
                dataList.add(read.next());
            }
            byte[] csvGz;
            if (dataList.size() > 1) {
                // в реальности пока сюда никто попадать не должен
                csvGz = new byte[dataList.stream().mapToInt(d -> d.getData().length).sum()];
                int index = 0;
                for (FeedsErrorsData fed : dataList) {
                    System.arraycopy(fed.getData(), 0, csvGz, index, fed.getData().length);
                    index += fed.getData().length;
                }
            } else if (!dataList.isEmpty()) {
                csvGz = dataList.get(0).getData();
            } else {
                return result;
            }
            CSVReader csvReader = new CSVReader(new InputStreamReader(GzipUtils.unGzip(new ByteArrayInputStream(csvGz))));
            boolean firstRow = true;
            for (String[] fields : csvReader) {
                if (firstRow) {
                    firstRow = false;
                    continue;
                }
                // temp workaround for severity + line
                if (fields.length == 6) {
                    final Matcher matcher = LEGACY_SEVERITY_LINE_PATTERN.matcher(fields[1]);
                    if (!matcher.matches()) {
                        continue;
                    }
                    result.add(new FeedsAllErrorsCsvRow(feedInfo.getUrl(), fields[0], matcher.group(1),
                            Optional.ofNullable(Strings.emptyToNull(matcher.group(2))).map(Integer::parseInt).orElse(null),
                            Optional.ofNullable(fields[2]).map(Integer::parseInt).orElse(null),
                            fields[3], fields[4], fields[5]));
                } else {
                    result.add(new FeedsAllErrorsCsvRow(feedInfo.getUrl(), fields[0], fields[1],
                            Optional.ofNullable(fields[2]).map(Integer::parseInt).orElse(null),
                            Optional.ofNullable(fields[3]).map(Integer::parseInt).orElse(null),
                            fields[4], fields[5], fields[6]));
                }
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return result;
    }

    private List<FeedsAllErrorsCsvRow> convertErrors(FeedRecord feedRecord) {
        return feedRecord.getErrors().values().stream().flatMap(Collection::stream).map(offerErrorInfo -> {
            String code = FeedsValidationErrorEnum.byCode(offerErrorInfo.getCode()).toString();
            // collect all details to simple json
            StringBuilder details = new StringBuilder();
            for (var iterator = ((ObjectNode) offerErrorInfo.getDetails()).fields(); iterator.hasNext(); ) {
                var entry = iterator.next();
                details.append(entry.getKey()).append("=").append(entry.getValue()).append(", ");
            }
            details.setLength(details.length() - 2);
            return new FeedsAllErrorsCsvRow(feedRecord.getUrl(), code, offerErrorInfo.getSeverity().getCode(), null, null, null,
                    offerErrorInfo.getMessage() + " " + details, null);
        }).collect(Collectors.toList());
    }

    @Override
    public Class<FeedsAllErrorsCsvRow> getRowClass() {
        return FeedsAllErrorsCsvRow.class;
    }
}
