package ru.yandex.webmaster3.worker.feeds.download;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.feeds.feed.FeedsValidationErrorEnum;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedInfo2;
import ru.yandex.webmaster3.core.feeds.feed.NativeFeedType;
import ru.yandex.webmaster3.core.util.GzipUtils;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.core.util.functional.ThrowingConsumer;
import ru.yandex.webmaster3.storage.download.common.DownloadFileType;
import ru.yandex.webmaster3.storage.download.common.MdsExportTaskData;
import ru.yandex.webmaster3.storage.feeds.FeedsNative2YDao;
import ru.yandex.webmaster3.storage.feeds.FeedsService;
import ru.yandex.webmaster3.storage.feeds.download.FeedsErrorsMdsExportDescriptor;
import ru.yandex.webmaster3.storage.feeds.logs.FeedsOffersLogsHistoryCHDao.FeedRecord;
import ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableRange;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.download.AbstractMdsDataProvider;

import static ru.yandex.webmaster3.core.util.functional.ThrowingConsumer.rethrowingUnchecked;

/**
 * Created by Oleg Bazdyrev on 18/01/2022.
 */
@Service
@AllArgsConstructor(onConstructor_ = @Autowired)
public class FeedsErrorsDownloadMdsDataProvider extends AbstractMdsDataProvider<FeedsErrorsData> {

    private static final String HEADER = "\"code\",\"severity\",\"line\",\"column\",\"url\",\"message\",\"context\"\n";
    private static final byte[] EMPTY_ARCHIVE = GzipUtils.compress(HEADER.getBytes(StandardCharsets.UTF_8));

    private final FeedsNative2YDao feedsNativeYDao;
    private final FeedsService feedsService;
    private final GoodsOffersLogsHistoryCHDao goodsOffersLogsHistoryCHDao;
    private final YtService ytService;
    @Value("hahn://home/webmaster/prod/feeds/offers/archive")
    private final YtPath feedsOffersArchive;
    @Value("hahn://home/webmaster/prod/feeds/serpdata/archive")
    private final YtPath feedsSerpdataArchive;

    @Override
    public void provide(MdsExportTaskData data, ThrowingConsumer<FeedsErrorsData, Exception> consumer) throws Exception {
        FeedsErrorsMdsExportDescriptor descriptor = (FeedsErrorsMdsExportDescriptor) data.getDescriptor();
        String feedUrl = Strings.isNullOrEmpty(descriptor.getFeed()) ? "" : feedsService.parseUrl(descriptor.getFeed()).url();

        boolean isFeedsValidationErrors = descriptor.getType() == DownloadFileType.FEEDS_VALIDATION_ERRORS;
        if (isFeedsValidationErrors) {
            NativeFeedInfo2 info = feedsNativeYDao.get(descriptor.getDomain(), feedUrl);
            if (info.getType() == NativeFeedType.STORES) {
                // read from clickhouse
                FeedRecord feedRecord = goodsOffersLogsHistoryCHDao.getHistory(descriptor.getDomain(),
                                Triple.of(info.getBusinessId(), info.getPartnerId(), info.getFeedId()),
                                new DateTime(descriptor.getTimestamp() * 1000L), new DateTime(descriptor.getTimestamp() * 1000L + 1)).stream()
                        .map(GoodsOffersLogsHistoryCHDao.GoodsFeedRecord::toFeedRecord).findFirst().orElse(null);
                if (feedRecord == null) {
                    rethrowingUnchecked(consumer).accept(new FeedsErrorsData("", "", 0L, 0L, 0L, EMPTY_ARCHIVE));
                    return;
                }
                StringBuilder sb = new StringBuilder(1024);
                // header
                sb.append(HEADER);
                feedRecord.getErrors().values().stream().flatMap(Collection::stream).forEach(offerErrorInfo -> {
                    String code = FeedsValidationErrorEnum.byCode(offerErrorInfo.getCode()).toString();
                    // collect all details to simple json
                    StringBuilder details = new StringBuilder();
                    for (var iterator = ((ObjectNode) offerErrorInfo.getDetails()).fields(); iterator.hasNext(); ) {
                        var entry = iterator.next();
                        details.append(entry.getKey()).append("=").append(entry.getValue()).append(", ");
                    }
                    details.setLength(details.length() - 2);
                    String message = offerErrorInfo.getMessage() + " " + details;
                    sb
                            .append(code).append(",")
                            .append(offerErrorInfo.getSeverity().getCode()).append(",")
                            .append(",") // line
                            .append(",") // column
                            .append(",") // url
                            .append(",\"").append(message.replaceAll("\"", "\\\"")).append("\",") //message
                            .append("\n"); // context
                });
                rethrowingUnchecked(consumer).accept(new FeedsErrorsData(
                        descriptor.getDomain(), descriptor.getFeed(), descriptor.getTimestamp(), descriptor.getTimestamp(),
                        0L, GzipUtils.compress(sb.toString().getBytes(StandardCharsets.UTF_8))));
                return;
            }
        }

        YtPath path = isFeedsValidationErrors ? feedsOffersArchive : feedsSerpdataArchive;
        String domain = isFeedsValidationErrors ? WwwUtil.cutWWWAndM(IdUtils.urlToHostId(feedUrl)) : descriptor.getDomain();
        ytService.withoutTransaction((cypressService) -> {
            AsyncTableReader<FeedsErrorsData> reader = new AsyncTableReader<>(
                    cypressService,
                    path,
                    YtTableRange.multiKey(domain, feedUrl, descriptor.getTimestamp()),
                    YtTableReadDriver.createYSONDriver(FeedsErrorsData.class)
            ).withRetry(3).splitInParts(10000L);

            try (AsyncTableReader.TableIterator<FeedsErrorsData> read = reader.read()) {
                if (!read.hasNext()) {
                    rethrowingUnchecked(consumer).accept(new FeedsErrorsData("", "", 0L, 0L, 0L, EMPTY_ARCHIVE));
                } else {
                    do {
                        rethrowingUnchecked(consumer).accept(read.next());
                    } while (read.hasNext());
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }

            return true;
        });
    }

    @Override
    public Class<FeedsErrorsData> getRowClass() {
        return FeedsErrorsData.class;
    }
}
