package ru.yandex.webmaster3.worker.feeds.logs;

import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.turbo.AbstractYqlPrepareImportTask;

import static ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao.F;
import static ru.yandex.webmaster3.storage.feeds.logs.GoodsOffersLogsHistoryCHDao.TABLE;

/**
 * Created by Oleg Bazdyrev on 16/12/2021.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ImportGoodsOffersLogsHistoryTask extends AbstractYqlPrepareImportTask {

    private static final int LINES_COUNT = 512;
    private static final Map<String, String> SOURCE_EXPRESSIONS = new ImmutableMap.Builder<String, String>()
            .put(F.BUSINESS_ID, "cast(businessId as String)")
            .put(F.PARTNER_ID, "cast(partnerId as String)")
            .put(F.FEED_ID, "cast(feedId as String)")
            .put(F.UPDATE_TIME, "cast(updateTime as String)")
            .put(F.ERRORS, "nvl(String::ReplaceAll(cast(Yson::SerializeJson(errors) as String), @@\\@@, @@\\\\@@), '')")
            .put(F.FEED_URL, "feedUrl")
            .put(F.STATS, "nvl(String::ReplaceAll(cast(Yson::SerializeJson(Yson::From(stats)) as String), @@\\@@, @@\\\\@@), '')")
            .put(F.STATUS, "status")
            .build();

    @Override
    protected CHTable getTable() {
        return TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.GOODS_OFFERS_LOGS_HISTORY;
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        String fields = getTable().getFields().stream().map(CHField::getName).map(SOURCE_EXPRESSIONS::get)
                .collect(Collectors.joining(" || '\\t' || ", "(", " || '\\n')"));

        YqlQueryBuilder yqlQueryBuilder = YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendText("PRAGMA yt.DefaultMemoryLimit = '4G';\n")
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM (\n")
                .appendText("SELECT (Digest::Fnv64(cast(businessId as String)) % " + getShardsCount() + ") as ShardId,")
                .appendText("((Digest::Fnv64(cast(feedId as String) || cast(updateTime as String))) % " + LINES_COUNT + ") as RowId,")
                .appendText(fields).appendText("as data ")
                .appendText("FROM ").appendTable(tablePath)
                .appendText("WHERE status != 'FEED_NOT_MODIFIED') \n GROUP BY ShardId, RowId;")
                .appendText("COMMIT;\n\n");

        return yqlQueryBuilder;
    }

    @Override
    protected String getTableSuffix(YtClickhouseDataLoad imprt) {
        return super.getTableSuffix(imprt);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_GOODS_OFFERS_LOGS_HISTORY;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.GOODS_OFFERS_LOGS_HISTORY;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 12/20 * * * *");
    }


}
