package ru.yandex.webmaster3.worker.feeds.statistics;

import java.util.Collections;
import java.util.Map;

import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.Duration;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.feeds.statistics.FeedsEnrichedShowsCHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.turbo.AbstractYqlPrepareImportTask;

/**
 * Created by Oleg Bazdyrev on 29/03/2022.
 */
public class ImportFeedsEnrichedShowsTask extends AbstractYqlPrepareImportTask {

    private static final DateTimeFormatter PARTITION_MONTH_FORMAT = DateTimeFormat.forPattern("yyyyMM");
    private static final String MERGE_TABLE_QUERY = "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} DELETE WHERE date = '${DATE}'\n\n" +
            "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} ATTACH PARTITION '${MONTH}' FROM ${DB}.${TEMP_TABLE}\n\n" +
            "" +
            "DROP TABLE ${DB}.${TEMP_TABLE};";

    @Override
    protected CHTable getTable() {
        return FeedsEnrichedShowsCHDao.TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.FEEDS_ENRICHED_SHOWS;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.FEEDS_ENRICHED_SHOWS;
    }

    @Override
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad imprt) throws Exception {
        return initByTableDateName(imprt);
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        String tableDate = imprt.getSourceTable().getName();
        YqlQueryBuilder yqlQueryBuilder = YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendFDefinition(YqlFunctions.ESCAPE)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("PRAGMA yt.DefaultMemoryLimit = '2G';\n\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM (\n")
                .appendText("SELECT (Digest::Fnv64(Domain) % " + getShardsCount() + ") as ShardId,")
                .appendText("((Digest::Fnv64(SetUrl) / " + getShardsCount() + ") % " + 128 + ") as RowId,")
                .appendText("('" + tableDate + "' || '\\t' ||\n")
                .appendText("Domain || '\\t' ||\n")
                .appendText("Type || '\\t' ||\n")
                .appendText("$escape(SetUrl) || '\\t' ||\n")
                .appendText("CAST(Shows as String) || '\\n') as data\n")
                .appendText("FROM").appendTable(imprt.getSourceTable()).appendText("\n")
                .appendText(") \n GROUP BY ShardId, RowId;")
                .appendText("COMMIT;\n\n");

        return yqlQueryBuilder;
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        String database = getTable().getDatabase();
        String tempTableName = getTable().replicatedMergeTreeTableName(-1, imprt.getData());
        // удаляем из общей таблицы подливаемую дату и вливаем свежие данные
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            boolean hasTable = !clickhouseSystemTablesCHDao.getTables(host, database, Collections.singleton(tempTableName)).isEmpty();
            if (!hasTable) {
                continue;
            }
            // получим минимальный и максимальный ts в новой таблице
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults().setHost(host).setTimeout(Duration.standardMinutes(10L));
            StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                    "DB", database,
                    "FULL_TABLE", FeedsEnrichedShowsCHDao.TABLE_NAME,
                    "TEMP_TABLE", tempTableName,
                    "DATE", imprt.getDateFrom().toString(),
                    "MONTH", PARTITION_MONTH_FORMAT.print(imprt.getDateFrom()),
                    "YEAR", String.valueOf(imprt.getDateFrom().getYear())
            ));
            for (String query : substitutor.replace(MERGE_TABLE_QUERY).split("\n\n")) {
                clickhouseServer.execute(ctx, query);
            }
        }

        return imprt.withNextState();
    }

    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        // no replication needed
        return imprt.withNextState();
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_FEEDS_ENRICHED_SHOWS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 38 * * * *");
    }

    public static void main(String[] args) {
        System.out.println(FeedsEnrichedShowsCHDao.TABLE.createReplicatedMergeTree(-1, ""));
    }
}
