package ru.yandex.webmaster3.worker.feeds.statistics;

import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.feeds.statistics.MarketOffersStatisticsYDao;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.settings.data.AbstractCommonDataState;
import ru.yandex.webmaster3.storage.util.ydb.YdbYqlService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 20/01/2022.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ImportMarketOffersStatisticsTask extends PeriodicTask<PeriodicTaskState> {

    private final MarketOffersStatisticsYDao marketOffersStatisticsYDao;
    private final SettingsService settingsService;
    @Value("hahn://home/market/production/indexer/datacamp/united/service_offers")
    private final YtPath serviceOffersPath;
    private final YdbYqlService ydbYqlService;

    public static final String DATA_SELECT_QUERY = """
            SELECT
                Date('${DATE}') as `date`,
                cast(business_id as Int64) as business_id,
                cast(partner_id as Int64) as partner_id,
                cast(feed_id as Int64) as feed_id,
                cast(count_if(meta.data_source == 5) as Int64) as offers,
                cast(count_if(meta.data_source == 24) as Int64) as robot_offers
            FROM ${SOURCE_TABLE}
            WHERE meta.rgb = 13
            GROUP BY business_id, shop_id as partner_id, nvl(meta.binded_feed_id.value, identifiers.real_feed_id, 0) as feed_id
            """;

    @Override
    public Result run(UUID runId) throws Exception {
        LocalDate today = LocalDate.now();
        LocalDate lastImport = Optional.ofNullable(settingsService.getSettingUncached(CommonDataType.LAST_MARKET_OFFERS_STATISTICS_IMPORT))
                .map(AbstractCommonDataState::getValue).map(LocalDate::parse).orElse(today.minusDays(1));
        if (today.isAfter(lastImport)) {
            StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                    "DATE", today.toString(),
                    "SOURCE_TABLE", serviceOffersPath.toYqlPath()
            ));
            ydbYqlService.importToYdb(marketOffersStatisticsYDao.getTablePath(), substitutor.replace(DATA_SELECT_QUERY),
                    "use " + serviceOffersPath.getCluster() + ";");
            settingsService.update(CommonDataType.LAST_MARKET_OFFERS_STATISTICS_IMPORT, today.toString());
        } else {
            log.info("Import already occurred today");
        }
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_MARKET_OFFERS_STATISTICS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 17 */4 * * *");
    }
}
