package ru.yandex.webmaster3.worker.fresh;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.checklist.dao.HostsWithUrlAlertYDao;
import ru.yandex.webmaster3.storage.checklist.dao.RealTimeSiteProblemsYDao;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;


/**
 * @author kravchenko99
 * @date 4/15/21
 */

@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class DisableRealtimeUrlAlertTask extends PeriodicTask<DisableRealtimeUrlAlertTask.State> {
    private final RealTimeSiteProblemsYDao realTimeSiteProblemsYDao;
    private final HostsWithUrlAlertYDao hostsWithUrlAlertYDao;

    @Setter
    private PeriodicTaskType taskType;
    @Setter
    private SiteProblemTypeEnum siteProblemTypeEnum;
    @Setter
    private SettingsService settingsService;

    @Override
    public Result run(UUID runId) throws Exception {
        CommonDataType type = CommonDataType.valueOf(getType().name());

        long currentHourInterval = TimeUtils.getCurrentHourInterval();

        CommonDataState settingUncached = settingsService.getSettingUncached(type);
        if (settingUncached != null && settingUncached.getValue().equals(String.valueOf(currentHourInterval))) {
            return Result.SUCCESS;
        }

        List<WebmasterHostId> hostIdsByProblemType = hostsWithUrlAlertYDao.getHostsByProblemType(siteProblemTypeEnum);
        List<WebmasterHostId> hostIdForDisableProblem = realTimeSiteProblemsYDao.listSitesProblems(hostIdsByProblemType,
                        siteProblemTypeEnum)
                .entrySet().stream()
                .filter(x -> x.getValue().getState() == SiteProblemState.PRESENT)
                .filter(x -> x.getValue().getContent() != null)
                .map(x -> Pair.of(x.getKey(), (SiteProblemContent.UrlAlertXxx) x.getValue().getContent()))
                .filter(x -> currentHourInterval - x.getValue().getHourInterval() >= 24)
                .map(Pair::getKey)
                .collect(Collectors.toList());


        Iterable<List<WebmasterHostId>> partition = Iterables.partition(hostIdForDisableProblem, 2500);
        for (var part: partition){
            realTimeSiteProblemsYDao.deleteProblems(part, List.of(siteProblemTypeEnum));
            hostsWithUrlAlertYDao.delete(part, siteProblemTypeEnum);
        }

        settingsService.update(type, String.valueOf(currentHourInterval));
        return Result.SUCCESS;
    }

    @Override
    public PeriodicTaskType getType() {
        return taskType;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 50 * * * *");
    }

    public static class State implements PeriodicTaskState {

    }
}
