package ru.yandex.webmaster3.worker.fresh;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.dao.HostsWithUrlAlertYDao;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.RealtimeUrlHttpCodesCHDao;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.RealtimeUrlHttpCodesExamplesCHDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author kravchenko99
 * @date 4/15/21
 */

@Slf4j
public abstract class EnableRealtimeUrlAlertTask extends PeriodicTask<EnableRealtimeUrlAlertTask.State> {

    @Autowired
    private RealtimeUrlHttpCodesCHDao realtimeUrlHttpCodesCHDao;
    @Autowired
    private ChecklistPageSamplesService checklistPageSamplesService;
    @Autowired
    private RealtimeUrlHttpCodesExamplesCHDao realtimeUrlHttpCodesExamplesCHDao;
    @Autowired
    private SiteProblemsService siteProblemsService;
    @Autowired
    protected AbtService abtService;
    @Autowired
    private AllVerifiedHostsCacheService allVerifiedHostsCacheService;
    @Autowired
    private SettingsService settingsService;
    @Autowired
    private HostsWithUrlAlertYDao hostsWithUrlAlertYDao;
    protected RealtimeUrlHttpCodesCHDao.Record.HttpGroup httpGroup;
    protected double threshold;
    protected ChecklistSamplesType samplesType;
    protected Experiment experiment;

    @Override
    public Result run(UUID runId) throws Exception {

        long currentHourInterval = TimeUtils.getCurrentHourInterval() - 1;
        log.info("hour_interval - {}", currentHourInterval);

        ProblemSignal problem = new ProblemSignal(createSiteProblemContent(currentHourInterval),
                new DateTime(currentHourInterval * TimeUtils.MILLIS_IN_HOUR));

        // находим все урлы которым нужно зажечь алерт, ожидается что таких не очень много порядка тысячи, если
        // больше скорее всего что-то у нас не так
        List<WebmasterHostId> hostIds = new ArrayList<>();
        // возможно фильтрацию стоит делать тоже в ch
        realtimeUrlHttpCodesCHDao.forEachStatistics(currentHourInterval, httpGroup, threshold, statRecord -> {
            WebmasterHostId hostId = statRecord.getHostId();
            //условие при котором зажигаем алерт
            if (!allVerifiedHostsCacheService.contains(hostId) || !abtService.isInHashExperiment(hostId, experiment)) {
                return;
            }
            hostIds.add(hostId);
        });

        log.info("hostId size - {}", hostIds.size());
        CommonDataType type = CommonDataType.valueOf(getType().name());
        CommonDataState settings = settingsService.getSettingUncached(type);
        // проверяем текущий hour interval и количество хостов, так как мог произойти затык и хосты посздно доехали
        String version = currentHourInterval + " " + hostIds.size();
        if (settings != null) {
            if (settings.getValue().equals(version)) {
                return Result.SUCCESS;
            }
        }


        Map<WebmasterHostId, List<RealtimeUrlHttpCodesExamplesCHDao.Record>> samples =
                realtimeUrlHttpCodesExamplesCHDao.getSamples(currentHourInterval, hostIds, httpGroup.getHttpCodeValue());

        Map<WebmasterHostId, List<String>> preparedSamples =
                samples.entrySet().stream().map(x ->
                        Pair.of(x.getKey(),
                                x.getValue().stream().map(JsonMapping::writeValueAsString).collect(Collectors.toList())
                        )
                ).collect(W3Collectors.toHashMap());
        checklistPageSamplesService.saveSamples(Map.of(samplesType, preparedSamples));


        hostsWithUrlAlertYDao.insert(hostIds, problem.getProblemType());
        for (var hostId : hostIds) {
//            log.info("Url with problem - {}", hostId);
            siteProblemsService.updateRealTimeProblem(hostId, problem);
        }
        settingsService.update(type, version);
        return Result.SUCCESS;
    }

    @NotNull
    protected abstract SiteProblemContent createSiteProblemContent(long currentHourInterval);

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }

    public static class State implements PeriodicTaskState {

    }
}
