package ru.yandex.webmaster3.worker.fresh;

import java.util.Optional;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.searchbase.SearchBaseUpdatesService;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * ishalaru
 * 09.03.2021
 **/
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component
public class RemoveOldFreshDataTask extends PeriodicTask<PeriodicTaskState> {
    @Qualifier("legacyMdbClickhouseServer")
    private final MdbClickhouseServer clickhouseServer;
    private final SearchBaseUpdatesService searchBaseUpdatesService;
    private final CommonDataStateYDao commonDataStateYDao;

    @Override
    public Result run(UUID runId) throws Exception {
        DateTime lastUpdateTime = Optional.ofNullable(commonDataStateYDao.getValue(CommonDataType.LAST_FRESH_TABLE_CLEARING))
                .map(state -> DateTime.parse(state.getValue()))
                .orElse(DateTime.parse("2021-01-01T00:00:00Z"));
        var curBaseDate = searchBaseUpdatesService.getSearchBaseUpdates().getCurrentBase().getBaseCollectionDate().toDateTime();
        if (curBaseDate.isAfter(lastUpdateTime)) {
            dropDataFromCH(curBaseDate);
            commonDataStateYDao.update(new CommonDataState(CommonDataType.LAST_FRESH_TABLE_CLEARING, curBaseDate.toString(), DateTime.now()));
        }

        return Result.SUCCESS;
    }

    private void dropDataFromCH(DateTime date) throws Exception {
        String query = String.format(
                "ALTER TABLE webmaster3_searchurls.fresh_url_event ON CLUSTER '{cluster}' DELETE WHERE base_date < %s",
                date.getMillis() / 1000
        );

        RetryUtils.execute(RetryUtils.linearBackoff(10, Duration.standardMinutes(10)), () -> {
            ClickhouseQueryContext.Builder chContextB = ClickhouseQueryContext.useDefaults()
                    .setHost(clickhouseServer.pickRandomAliveHost());
            clickhouseServer.execute(chContextB, query);
        });
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.REMOVE_OLD_FRESH_DATA;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("1 2 4 * * *");
    }
}
