package ru.yandex.webmaster3.worker.host;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableLong;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.host.dao.HostsYDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UploadNewWebmasterHostsTask extends PeriodicTask<PeriodicTaskState> {
    private static final String TABLE_PATH_PREFIX = "user/webmaster-new-hosts/hosts-";

    private final HostsYDao hostsYDao;
    private final SettingsService settingsService;
    private final YtService ytService;
    @Setter
    private YtPath tableDir;
    @Setter
    private String tmpFileName;

    @Override
    public Result run(UUID runId) throws Exception {
        DateTime startTime = new DateTime();
        File tmpFile = new File(tmpFileName);
        CommonDataState lastUploadedRec = settingsService.getSettingUncached(CommonDataType.LAST_UPLOADED_NEW_HOSTS);
        DateTime lastUploadedAddTime;
        if (lastUploadedRec == null) {
            lastUploadedAddTime = DateTime.now().minus(Duration.standardHours(1));
        } else {
            lastUploadedAddTime = new DateTime(Long.parseLong(lastUploadedRec.getValue()));
        }

        MutableLong newHosts = new MutableLong(0L);
        try (PrintWriter pw = new PrintWriter(tmpFile)) {
            hostsYDao.forEach(pair -> {
                DateTime addedDate = pair.getRight();
                if (addedDate != null && addedDate.isAfter(lastUploadedAddTime) && addedDate.isBefore(startTime)) {
                    pw.println(IdUtils.hostIdToUrl(pair.getLeft()));
                    newHosts.increment();
                }
            });
        }

        log.info("Found {} new hosts", newHosts.getValue());
        if (newHosts.getValue() == 0L) {
            return new Result(TaskResult.SUCCESS);
        }
        ytService.inTransaction(tableDir).execute(cypressService -> {
            YtPath tableName = YtPath.path(tableDir, TABLE_PATH_PREFIX + startTime.getMillis());
            if (cypressService.exists(tableName)) {
                cypressService.remove(tableName);
            }
            cypressService.create(tableName, YtNode.NodeType.TABLE, true);
            cypressService.writeTable(tableName, tableWriter -> {
                try (BufferedReader reader = new BufferedReader(new FileReader(tmpFile))) {
                    String line;
                    while ((line = reader.readLine()) != null) {
                        tableWriter.column("key", line);
                        tableWriter.column("value", "");
                        tableWriter.rowEnd();
                    }
                } catch (YtException | IOException e) {
                    throw new WebmasterException("Failed to upload webmaster hosts diff",
                            new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
                }
            });
            settingsService.update(CommonDataType.LAST_UPLOADED_NEW_HOSTS, String.valueOf(startTime.getMillis()));
            return true;
        });
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_NEW_WEBMASTER_HOSTS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never(); // not needed now, might be useful in the future
        // return TaskSchedule.startByCron("0 0 * * * *");
    }

}
