package ru.yandex.webmaster3.worker.host.moderation;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.mutable.MutableObject;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.regions.W3RegionsTreeService;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequest;
import ru.yandex.webmaster3.storage.host.moderation.regions.dao.HostRegionsModerationRequestsYDao;
import ru.yandex.webmaster3.storage.host.moderation.regions.service.HostRegionsModerationService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

import static ru.yandex.webmaster3.storage.host.moderation.regions.HostRegionsModerationRequestStatus.IN_MODERATION;

/**
 * @author leonidrom
 * <p>
 * Заново посылает на модерацию ассессорами заявки, которые по каким то причинам
 * так и не были ими обработаны.
 * В идеале, таких заявок быть не должно, но пока все причины продалбывания заявок
 * не устранены, используется такой вот костыль.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class HostRegionsModerationResendLostRequestsTask extends PeriodicTask<HostRegionsModerationResendLostRequestsTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(HostRegionsModerationResendLostRequestsTask.class);
    private static DateTime SKIP_REQUESTS_BEFORE_DATE = DateTime.parse("2018-01-25T19:54:52.128+03:00");
    private static int MAX_REQUEST_AGE_IN_DAYS = 14;

    private final HostRegionsModerationRequestsYDao hrmRequestsYDao;
    private final HostRegionsModerationService hostRegionsModerationService;
    private final W3RegionsTreeService w3regionsTreeService;
    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;

    @Override
    public Result run(UUID runId) throws Exception {
        TaskState ts = new TaskState();
        setState(ts);

        MutableObject<HostRegionsModerationRequest> maxReqObj = new MutableObject<>(null);
        MutableObject<WebmasterHostId> curHostIdObj = new MutableObject<>(null);
        hrmRequestsYDao.forEach(req -> {
            try {
                var hostId = req.getHostId();
                if (!allVerifiedHostsCacheService.contains(hostId)) {
                    return;
                }

                var curHostId = curHostIdObj.getValue();
                var maxReq = maxReqObj.getValue();
                if (!hostId.equals(curHostId)) {
                    if (maxReq != null) {
                        processReq(maxReq);
                    }

                    maxReqObj.setValue(req);
                    curHostIdObj.setValue(hostId);
                } else {
                    if (req.getUpdateDate().isAfter(maxReq.getUpdateDate())) {
                        maxReqObj.setValue(req);
                    }
                }

            } catch (Exception exp) {
                throw new WebmasterException("Ydb error",
                        new WebmasterErrorResponse.YDBErrorResponse(getClass(), exp), exp);
            }
        });

        var maxReq = maxReqObj.getValue();
        if (maxReq != null) {
            processReq(maxReq);
        }

        return new Result(TaskResult.SUCCESS);
    }

    private void processReq(HostRegionsModerationRequest req) {
        if (this.state.totalResend >= 1000) {
            return;
        }

        // узнаем есть ли потерянный запрос для данного хоста
        boolean isDone = req.getStatus() != IN_MODERATION;
        long reqAgeInDays = TimeUnit.MILLISECONDS.toDays(System.currentTimeMillis() - req.getUpdateDate().getMillis());

        if (isDone || reqAgeInDays < MAX_REQUEST_AGE_IN_DAYS || SKIP_REQUESTS_BEFORE_DATE.isAfter(req.getUpdateDate())) {
            return;
        }

        // отфильтруем заявки с неизвестными регионами
        for (int regionId : req.getRequestedRegions()) {
            if (w3regionsTreeService.getExactRegionInfo(regionId) == null) {
                hostRegionsModerationService.cancelModerationRequest(req.getHostId(), req.getRequestId());
                return;
            }
        }

        // и заново создадим его
        log.info("Resending {}", req);
        hostRegionsModerationService.createModerationRequest(req.getHostId(),
                req.getRequestedRegions(), req.getEvidenceUrl(), false);

        this.state.totalResend++;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.HOST_REGIONS_MODERATION_RESEND_LOST_REQUESTS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 0 * * *");
    }

    public static class TaskState implements PeriodicTaskState {
        int totalResend;

        public int getTotalResend() {
            return totalResend;
        }
    }
}
