package ru.yandex.webmaster3.worker.host.verification;

import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

import com.datastax.driver.core.utils.UUIDs;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.core.worker.task.VerifyHostTaskData;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskPriority;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RecheckVerificationsTask extends PeriodicTask<PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(RecheckVerificationsTask.class);

    private final UserHostsService userHostsService;
    private final UserHostVerificationYDao userHostVerificationYDao;
    private final WorkerClient workerClient;

    @Override
    public Result run(UUID runId) throws Exception {
        userHostsService.forEach(
                pair -> {
                    long userId = pair.getLeft();
                    UserVerifiedHost userVerifiedHost = pair.getRight();
                    if (userVerifiedHost.getVerificationType() == VerificationType.UNKNOWN ||
                            userVerifiedHost.getVerifiedUntilDate() == null ||
                            userVerifiedHost.getVerifiedUntilDate().isBeforeNow()) {
                        log.info("Reverifying user {} host {} type {}", userId, userVerifiedHost.getWebmasterHostId(),
                                userVerifiedHost.getVerificationType());
                        long uin = userVerifiedHost.getVerificationUin();
                        if (userVerifiedHost.getVerificationType() == VerificationType.UNKNOWN && uin == 0L) {
                            uin = ThreadLocalRandom.current().nextLong();
                        }
                        UUID recordId = UUIDs.timeBased();
                        userHostVerificationYDao.addVerificationRecord(
                                UserHostVerificationInfo.createVerificationStartRecord(
                                        recordId,
                                        userId,
                                        userVerifiedHost.getWebmasterHostId(),
                                        userVerifiedHost.getVerificationType(),
                                        uin,
                                        VerificationCausedBy.REGULAR_RECHECK
                                )
                        );
                        workerClient.enqueueTask(VerifyHostTaskData.createTaskData(
                                userVerifiedHost.getWebmasterHostId(),
                                userId,
                                recordId,
                                userVerifiedHost.getVerificationType(),
                                WorkerTaskPriority.NORMAL
                        ));
                    }
                }
        );
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.RECHECK_VERIFICATIONS_TASK;
    }

    @Override
    public TaskSchedule getSchedule() {
        // WMC-11875
        return TaskSchedule.startByCron("0 0 2 * * *");
    }
}
