package ru.yandex.webmaster3.worker.host.verification;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.addurl.AddUrlService;
import ru.yandex.webmaster3.core.antispam.threats.CheckForHostThreatsTaskData;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.storage.mobile.HostMobileAuditService;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserVerifiesHostEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsObserver;

/**
 * @author leonidrom
 * <p>
 * Всякие действия над хостом после его подтверждения
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UserVerifiesHostEventPostprocessingObserver implements WMCEventsObserver {
    //TODO: remove this class after 2021.01.13
    private final DateTime PROCESSING_DATE_TIME = new DateTime(2021, 01, 13, 0, 0);
    private final WorkerClient workerClient;
    private final AddUrlService addUrlService;
    private final HostMobileAuditService hostMobileAuditService;

    @Override
    public boolean observe(WMCEvent event) {
        if (!(event.getContent() instanceof UserVerifiesHostEvent)) {
            log.error("Unexpected event type: {}", event.getContent().getType());
            return false;
        }

        UserVerifiesHostEvent content = (UserVerifiesHostEvent) event.getContent();
        // нас интересуют только новые хосты, которых раньше не было
        if (content.getHostId() == null || !content.isNewHost()) {
            return false;
        }
        if (DateTime.now().isAfter(PROCESSING_DATE_TIME)) {
            return false;
        }
        WebmasterHostId hostId = content.getHostId();
        log.info("Processing new verified host: {}", hostId);

        // проверим хост на угрозы
        workerClient.enqueueTask(new CheckForHostThreatsTaskData(hostId));

        // переобойдем его морду
        addUrlService.requestRecrawl(IdUtils.hostIdToReadableUrl(hostId) + "/");

        // и проверим на мобилопригодность
        hostMobileAuditService.recheck(hostId);

        return true;
    }
}
