package ru.yandex.webmaster3.worker.hoststat.download;

import java.util.EnumMap;
import java.util.Map;

import NWebmaster.NProto.Hostinfo;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.functional.ThrowingConsumer;
import ru.yandex.webmaster3.storage.download.common.MdsExportTaskData;
import ru.yandex.webmaster3.storage.hoststat.download.ContentAttrSamplesMdsExportDescriptor;
import ru.yandex.webmaster3.storage.hoststat.download.ContentAttrSamplesMdsExportDescriptor.ContentAttrType;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableRange;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.download.AbstractMdsDataProvider;

/**
 * ishalaru
 * 27.03.2020
 **/
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component
public class ContentAttrSamplesMdsDataProvider extends AbstractMdsDataProvider<ContentAttrSampleCsvRow> {

    private static final EnumMap<ContentAttrType, String> TABLE_NAMES = new EnumMap<>(Map.of(
            ContentAttrType.TITLE, "titles",
            ContentAttrType.DESCRIPTION, "descriptions"
    ));

    private final YtService ytService;

    @Value("${external.yt.service.arnold}://home/webmaster/prod/stage/export/hostinfo")
    private YtPath rootDir;

    @Override
    public void provide(MdsExportTaskData exportTaskData, ThrowingConsumer<ContentAttrSampleCsvRow, Exception> consumer) throws Exception {
        var exportDescriptor = (ContentAttrSamplesMdsExportDescriptor) exportTaskData.getDescriptor();
        YtPath sourceTable = YtPath.path(rootDir, TABLE_NAMES.get(exportDescriptor.getContentAttrType()));
        ytService.withoutTransaction(cypressService -> {
            AsyncTableReader<ContentAttrRecord> tableReader = new AsyncTableReader<>(cypressService, sourceTable,
                    YtTableRange.singleKey(exportDescriptor.getHost()), YtTableReadDriver.createYSONDriver(ContentAttrRecord.class))
                    .withRetry(3)
                    .withThreadName("host-content-attr-reader")
                    .splitInParts(1000L);

            try (var iterator = tableReader.read()) {
                while (iterator.hasNext()) {
                    ContentAttrRecord record = iterator.next();
                    if (record.getCount() == null || record.getCount() == 0) {
                        // fake record
                        continue;
                    }
                    Hostinfo.THostContentAttrSamples contentAttrSamples = Hostinfo.THostContentAttrSamples.parseFrom(record.getSamples());
                    // generating records
                    consumer.accept(new ContentAttrSampleCsvRow(record.getValue(), record.getCount(), null, null));
                    for (Hostinfo.TPageSample pageSample : contentAttrSamples.getPageSamplesList()) {
                        String lastAccess = pageSample.getLastAccess() == 0L ? null : new LocalDate(pageSample.getLastAccess() * 1000L).toString();
                        consumer.accept(new ContentAttrSampleCsvRow(null, null, pageSample.getPath(), lastAccess));
                    }
                    // space
                    consumer.accept(new ContentAttrSampleCsvRow(null, null, null, null));
                }
            } catch (Exception e) {
                throw new WebmasterException("Error when reading content attrs from YT",
                        new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            }
            return true;
        });
    }

    @Override
    public Class<ContentAttrSampleCsvRow> getRowClass() {
        return ContentAttrSampleCsvRow.class;
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ContentAttrRecord {
        @JsonProperty("Host")
        String host;
        @JsonProperty("Value")
        String value;
        @JsonProperty("Count")
        Long count;
        @JsonProperty("Samples")
        byte[] samples;
    }
}
