package ru.yandex.webmaster3.worker.http;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import lombok.AllArgsConstructor;
import lombok.Getter;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ActionStatus;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.storage.digest.DigestAcceptance;
import ru.yandex.webmaster3.storage.digest.DigestAcceptanceYDao;
import ru.yandex.webmaster3.storage.digest.DigestMessagesCHDao;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.searchquery.importing.dao.YtClickhouseDataLoadYDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.settings.data.AbstractCommonDataState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;

/**
 * @author shabashoff
 */
@ReadAction
@Component("/digest/info")
@AllArgsConstructor(onConstructor_ = @Autowired)
public class GetDigestInfoAction extends Action<GetDigestInfoAction.Req, GetDigestInfoAction.Res> {
    private final DigestMessagesCHDao digestMessagesCHDao;
    private final DigestAcceptanceYDao digestAcceptanceYDao;
    private final YtClickhouseDataLoadYDao ytClickhouseDataLoadCDao;
    private final SettingsService settingsService;
    @Value("${webmaster3.worker.digest.templateHosts}")
    private final String[] templateHosts;

    @Override
    public Res process(Req request) throws WebmasterException {
        LocalDate lastSentDigest = Optional.ofNullable(settingsService.getSettingUncached(CommonDataType.LAST_SENT_DIGEST_DATE))
                                            .map(CommonDataState::getValue)
                                            .map(LocalDate::parse)
                                            .orElse(null);
        String customEnBlock = Optional.ofNullable(settingsService.getSettingUncached(CommonDataType.DIGEST_CUSTOM_BLOCK_EN))
                                        .map(AbstractCommonDataState::getValue)
                                        .orElse(null);
        String customRuBlock = Optional.ofNullable(settingsService.getSettingUncached(CommonDataType.DIGEST_CUSTOM_BLOCK_RU))
                                        .map(AbstractCommonDataState::getValue)
                                        .orElse(null);

        YtClickhouseDataLoad tableDataInfo = ytClickhouseDataLoadCDao.load(YtClickhouseDataLoadType.DIGEST_MESSAGES);
        Optional<DigestAcceptance> acceptance = digestAcceptanceYDao.getAcceptanceByDate(tableDataInfo.getDateTo());

        Set<LocalDate> tableDates = digestMessagesCHDao.getTableDates();
        tableDates.add(lastSentDigest);
        List<LocalDate> tableDatesList = new ArrayList<>(tableDates);
        tableDatesList.sort(null);

        return new Res(
                tableDatesList,
                lastSentDigest,
                tableDataInfo,
                acceptance.orElse(null),
                templateHosts,
                customEnBlock,
                customRuBlock
        );
    }

    public static class Req implements ActionRequest {
    }

    @Getter
    @AllArgsConstructor
    public static class Res implements ActionResponse {
        private final List<LocalDate> digestDates;
        private final LocalDate lastSentDigest;
        private final YtClickhouseDataLoad importDigestTask;
        private final DigestAcceptance digestAcceptance;
        private final String[] digestTemplates;
        private final String customEnBlock, customRuBlock;

        @Override
        public ActionStatus getRequestStatus() {
            return ActionStatus.SUCCESS;
        }
    }
}
