package ru.yandex.webmaster3.worker.http.abt;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.util.UriComponentsBuilder;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.conductor.ConductorClient;
import ru.yandex.webmaster3.core.util.conductor.ConductorHostInfo;

import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author leonidrom
 */
@Component("/experiment/cleanCache")
@Slf4j
public class CleanExperimentsCacheAction extends Action<CleanExperimentsCacheAction.Request, CleanExperimentsCacheAction.Response> {
    private static final int DEFAULT_VIEWER_PORT = 33585;
    private static String ACTION_PATH = "/abt/cleanCache.json";

    private final ConductorClient conductorClient;
    private final String viewerConductorGroup;
    private final CloseableHttpClient httpClient = HttpClients.createDefault();

    @Autowired
    public CleanExperimentsCacheAction(
            ConductorClient conductorClient,
            @Value("${webmaster3.core.conductorClients.viewersGroup}") String viewerConductorGroup) {
        this.conductorClient = conductorClient;
        this.viewerConductorGroup = viewerConductorGroup;
    }

    @Override
    public Response process(Request request) throws WebmasterException {
        List<ConductorHostInfo> hostInfos = conductorClient.listHostsInGroup(viewerConductorGroup);
        List<URI> hosts = hostInfos.stream()
                .map(hostInfo -> URI.create("http://" + hostInfo.getHostName()))
                .collect(Collectors.toList());
        for (var hostUri : hosts) {
            cleanCacheForHost(hostUri);
        }

        return new Response();
    }

    private void cleanCacheForHost(URI hostUri) {
        log.info("Cleaning cache for {}", hostUri);

        URI requestUri = UriComponentsBuilder.fromUri(hostUri)
                .port(DEFAULT_VIEWER_PORT)
                .path(ACTION_PATH)
                .build()
                .toUri();

        HttpGet req = new HttpGet(requestUri);
        try (CloseableHttpResponse httpResponse = httpClient.execute(req)) {
            if (httpResponse.getStatusLine().getStatusCode() != 200) {
                throw new WebmasterException("Failed to clean cache for host " + hostUri,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), ""));
            }
        } catch (IOException e) {
            log.error("Failed to clean cache for host " + hostUri, e);
            // игнорируем
        }
    }

    public static class Request implements ActionRequest {}
    public static class Response implements ActionResponse.NormalResponse {}
}
