package ru.yandex.webmaster3.worker.http.clickhouse;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.SneakyThrows;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.ytimport.dao.YtClickhouseImportQueueYDao;
import ru.yandex.webmaster3.worker.http.clickhouse.CancelClickhouseYtImportTaskAction.Request;
import ru.yandex.webmaster3.worker.http.clickhouse.CancelClickhouseYtImportTaskAction.Response;

/**
 * Created by Oleg Bazdyrev on 20/04/2017.
 */
@WriteAction
@Description("Отменяет выполнение задачи импорта YT->Clickhouse")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class CancelClickhouseYtImportTaskAction extends Action<Request, Response> {

    private final YtClickhouseImportQueueYDao ytClickhouseImportQueueYDao;
    private final YtService ytService;

    @Value("${external.yt.service.locke.root.default}/importer/legacy-tasks")
    private YtPath tasksBasePath;

    @Override
    @SneakyThrows
    public Response process(Request request) throws WebmasterException {

        UUID taskId = UUID.fromString(request.id);
        YtClickhouseImportQueueYDao.YtClickhouseImportRecord task = ytClickhouseImportQueueYDao.getTask(taskId);
        if (task == null && !request.skipCheckId) {
            throw new IllegalStateException("Task with id " + request.id + " doesn't exist");
        }
        List<String> droppedTables = new ArrayList<>();
        ytClickhouseImportQueueYDao.deleteTask(taskId);
        YtPath path = YtPath.path(tasksBasePath, request.cluster);
        ytService.withoutTransaction(cypressService -> {
            cypressService.list(path).stream()
                    .map(cypressService::list)
                    .flatMap(Collection::stream)
                    .filter(node -> request.id.equals(node.getName().split("_")[1]))
                    .forEach(node -> {
                        droppedTables.add(node.getPathWithoutCluster());
                        cypressService.remove(node);
                    });
            return true;
        });
        return new Response(droppedTables);
    }


    @Setter(onMethod_ = @RequestQueryProperty(required = true))
    public static class Request implements ActionRequest {
        String cluster;
        String id;
        // если вдруг удалили таску но не удалили ноды на locke
        @Setter(onMethod_ = @RequestQueryProperty(required = false))
        boolean skipCheckId;
    }

    @lombok.Value
    public static class Response implements ActionResponse.NormalResponse {
        List<String> droppedTables;
    }
}
