package ru.yandex.webmaster3.worker.http.clickhouse;

import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.worker.http.clickhouse.CreateClickhouseReplicasAction.Request;
import ru.yandex.webmaster3.worker.http.clickhouse.CreateClickhouseReplicasAction.Response;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.*;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

import java.util.*;

/**
 * Created by Oleg Bazdyrev on 02/06/2017.
 */
@WriteAction
@Description("Ручка для создания недостающих реплик")
public class CreateClickhouseReplicasAction extends Action<Request, Response> {

    private ClickhouseServer clickhouseServer;

    @Override
    public Response process(Request request) throws WebmasterException {
        for (String table : request.tables) {
            createMissingReplicas(clickhouseServer, request.database, table);
        }
        return new Response();
    }

    public static void createMissingReplicas(ClickhouseServer clickhouseServer, String database, String table) {
        // ищем. где таблица уже создана и делаем копии
        Map<Integer, String> createStatementByShard = new HashMap<>();
        List<ClickhouseHost> hostForCreateTable = new ArrayList<>();
        Boolean sharded = null;
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            try {
                ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults().setHost(host);
                String query = "show create table " + database + "." + table;
                String statement = clickhouseServer.queryOne(context, query, row -> row.getString("statement")).get();
                boolean localSharded = statement.contains("{shard}");
                Preconditions.checkState(sharded == null || sharded == localSharded);
                sharded = localSharded;
                createStatementByShard.put(sharded ? host.getShard() : - 1, statement);
            } catch (ClickhouseException e) {
                // table not exists
                hostForCreateTable.add(host);
            }
        }
        // creating tables
        for (ClickhouseHost host : hostForCreateTable) {
            String statement = createStatementByShard.get(sharded ? host.getShard() : -1);
            if (statement == null) {
                continue;
                //throw new WebmasterException("Could not get CREATE TABLE statement for shard " + host.getShard(),
                //        new WebmasterErrorResponse.ClickhouseErrorResponse(CreateClickhouseReplicasAction.class, null));
            }
            ClickhouseQueryContext.Builder context = ClickhouseQueryContext.useDefaults().setHost(host);
            try {
                clickhouseServer.execute(context, ClickhouseServer.QueryType.INSERT, statement,
                        Optional.empty(), Optional.empty());
            } catch (ClickhouseException e) {
                throw new WebmasterException("Could not create table",
                        new WebmasterErrorResponse.ClickhouseErrorResponse(CreateClickhouseReplicasAction.class, statement, e), e);
            }
        }
    }

    @Required
    public void setClickhouseServer(ClickhouseServer clickhouseServer) {
        this.clickhouseServer = clickhouseServer;
    }

    public static class Request implements ActionRequest {

        private String database;
        private List<String> tables;

        @RequestQueryProperty(required = true)
        @Description("БД, в которой требуется создать таблицы")
        public void setDatabase(String database) {
            this.database = database;
        }

        @RequestQueryProperty(required = true)
        @Description("Список таблиц, которые необходимо отреплицировать")
        public void setTables(List<String> tables) {
            this.tables = tables;
        }

    }

    public static class Response implements ActionResponse.NormalResponse {

    }
}
