package ru.yandex.webmaster3.worker.iks;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.achievements.dao.HostAchievementsRivalsYDao;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * ishalaru
 * 28.09.2020
 **/
@Slf4j
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ExportIksRivalsPeriodicTask extends PeriodicTask<ExportIksRivalsPeriodicTask.State> {
    private final int BATCH_SIZE = 4000;
    @Value("${external.yt.service.arnold.root.default}/export/iks_rival")
    private YtPath path;
    private final YtService ytService;
    private final HostAchievementsRivalsYDao hostAchievementsRivalsYDao;

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> HOST_ID = SCHEMA.addColumn("host_id", YtColumn.Type.STRING);
        YtColumn<String> RIVAL = SCHEMA.addColumn("rival", YtColumn.Type.STRING);
    }

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new State());


        YtTableData tableData = ytService.prepareTableData("export-iks-rivals", tableWriter -> {
                    try {
                        List<HostAchievementsRivalsYDao.RowRecord> batch = new ArrayList<>();
                        hostAchievementsRivalsYDao.forEach(rowRecord -> {
                            batch.add(rowRecord);
                            if (batch.size() > BATCH_SIZE) {
                                writeBatch(batch, tableWriter);
                                batch.clear();
                            }
                        });
                        if (!batch.isEmpty()) {
                            writeBatch(batch, tableWriter);
                        }
                    } catch (YtException e) {
                        throw new WebmasterException("YT error",
                                new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                    }
                }
        );
        ytService.inTransaction(path).execute(cService -> {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
            if (cService.exists(path)) {
                cService.remove(path);
            }

            cService.create(path, YtNode.NodeType.TABLE, true, attributes);
            cService.writeTable(path, tableData);

            return true;
        });
        return Result.SUCCESS;
    }

    private void writeBatch(List<HostAchievementsRivalsYDao.RowRecord> batch, TableWriter tableWriter) {
        for (HostAchievementsRivalsYDao.RowRecord rowRecord : batch) {
            F.HOST_ID.set(tableWriter, rowRecord.getHostId().toString());
            F.RIVAL.set(tableWriter, rowRecord.getRival());
            tableWriter.rowEnd();
        }
    }


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_IKS_RIVALS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 43 * * * *");
    }

    public static class State implements PeriodicTaskState {
    }
}
