package ru.yandex.webmaster3.worker.iks;

import java.util.Collections;
import java.util.Map;

import lombok.Setter;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.Duration;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.storage.iks.dao.IksUpdateDateYDao;
import ru.yandex.webmaster3.storage.iks.dao.MdbIksCHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 18/03/2021.
 */
public class MdbImportIksHistoryTask extends MdbImportIksTask {

    private static final String MERGE_TABLE_QUERY = "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} DELETE WHERE date = '${DATE}';\n\n" +
            "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} ATTACH PARTITION '${YEAR}' FROM ${DB}.${TEMP_TABLE}\n\n" +
            "" +
            "DROP TABLE ${DB}.${TEMP_TABLE};";

    @Setter
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;
    @Setter
    private IksUpdateDateYDao iksUpdateDateYDao;
    @Setter
    private IksService iksService;

    @Override
    protected int getShardsCount() {
        return clickhouseServer.getShardsCount();
    }

    @Override
    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        String database = getTable().getDatabase();
        String tempTableName = getTable().replicatedMergeTreeTableName(-1, imprt.getData());
        // удаляем из общей таблицы подливаемую дату и вливаем свежие данные
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            boolean hasTable = !clickhouseSystemTablesCHDao.getTables(host, database, Collections.singleton(tempTableName)).isEmpty();
            if (!hasTable) {
                continue;
            }
            // получим минимальный и максимальный ts в новой таблице
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults().setHost(host).setTimeout(Duration.standardMinutes(10L));
            StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                    "DB", database,
                    "FULL_TABLE", MdbIksCHDao.HISTORY_TABLE_NAME,
                    "TEMP_TABLE", tempTableName,
                    "DATE", imprt.getDateFrom().toString(),
                    "YEAR", String.valueOf(imprt.getDateFrom().getYear())
            ));
            for (String query : substitutor.replace(MERGE_TABLE_QUERY).split("\n\n")) {
                clickhouseServer.execute(ctx, query);
            }
        }
        iksUpdateDateYDao.insert(imprt.getDateTo().toDateTimeAtStartOfDay());
        iksService.updateCurrentIksLastImportDate(imprt.getDateTo().toDateTimeAtStartOfDay());
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    @Override
    protected CHTable getTable() {
        return MdbIksCHDao.HISTORY_TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.IKS_HISTORY;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.IKS_HISTORY;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.MDB_IMPORT_IKS_HISTORY;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0/30 * * * *");
    }
}
