package ru.yandex.webmaster3.worker.importanturls;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Repository;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.importanturls.HostsWithImportantUrlsYDao;
import ru.yandex.webmaster3.storage.importanturls.ImportantUrlsYDao;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlRequest;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.spam.ISpamHostFilter;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableRange;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Repository("autoAddImportantUrlsTask")
@Slf4j
public class AutoAddImportantUrlsTask extends PeriodicTask<AutoAddImportantUrlsTask.TaskState> {
    private final static long SECONDS_AFTER_VERIFICATION = Duration.ofDays(7).toSeconds();
    private static final DateTimeFormatter DATE_FORMAT_IN_NODE_NAME = DateTimeFormat.forPattern("yyyyMMdd");

    private final HostsWithImportantUrlsYDao hostsWithImportantUrlsYDao;
    private final ImportantUrlsYDao importantUrlsYDao;
    private final WMCEventsService wmcEventsService;
    private final YqlService yqlService;
    private final YtService ytService;
    private final ISpamHostFilter fastSpamHostFilter;
    private final JupiterUtils jupiterUtils;

    private TaskState state;

    @Value("${webmaster3.worker.yt.importantUrls.hosts-with-important-urls.path}")
    private YtPath hostsWithImportantUrlsTable;
    @Value("${webmaster3.worker.yt.importantUrls.updates.path}")
    private YtPath updatesTable;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.arnold.export.archive.verifiedHosts.path}")
    private YtPath arnoldVerifiedHostsExportPath;
    @Value("${webmaster3.worker.yt.searchqueries.top_urls.path}")
    private YtPath topUrlsTable;
    @Value("${webmaster3.worker.yt.importantUrls.tmp.top-urls-for-add-to-important-urls.path}")
    private YtPath tmpTable;
    @Value("${webmaster3.worker.yt.sitetree.search.history-full-new-gone}")
    private YtPath historyFullNewGone;

    private String createYqlQuery() {

        final String verifiedHostsTableName =
                "webmaster-verified-hosts." + DATE_FORMAT_IN_NODE_NAME.print(DateTime.now()/*.minusDays(1)*/);
        YtPath verifiedHostsTable = YtPath.path(arnoldVerifiedHostsExportPath, verifiedHostsTableName);

        final YtPath jupiterCurrentPathExport = jupiterUtils.getCurrentPathExport();
        final YtPath mirrosPath = YtPath.path(jupiterCurrentPathExport, "mirrors/mirrors");

        final Map<String, String> values = Map.of(
                "hostsWithImportantUrlsTable", hostsWithImportantUrlsTable.toYqlPath(),
                "updatesTable", updatesTable.toYqlPath(),
                "arnoldVerifiedHostsExportPath", arnoldVerifiedHostsExportPath.toYqlPath(),
                "topUrlsTable", topUrlsTable.toYqlPath(),
                "tmpTable", tmpTable.toYqlPath(),
                "historyFullNewGone", historyFullNewGone.toYqlPath(),
                "SECONDS_AFTER_VERIFICATION", Long.toString(SECONDS_AFTER_VERIFICATION),
                "verifiedHostsTable", verifiedHostsTable.toYqlPath(),
                "mirrosPath", mirrosPath.toYqlPath()
        );
        StrSubstitutor sub = new StrSubstitutor(values);

        String template = "$secondsBetweenVerificationAndAutoAdd=${SECONDS_AFTER_VERIFICATION};\n\n" +
                "$ninetyDaysInSeconds=90 * 24 * 60 * 60;\n\n" +

                "$historyFor90days = SELECT Host, Path, count(*) AS cnt FROM ${historyFullNewGone} " +
                "WHERE -`Timestamp` >= DateTime::ToSeconds(CurrentUtcTimestamp(1)) - $ninetyDaysInSeconds GROUP BY " +
                "Host, Path;\n\n" +

                "$hosts_with_important_urls = SELECT host_id FROM ${hostsWithImportantUrlsTable};\n\n" +
                "$important_urls_updates = SELECT DISTINCT `Host` FROM ${updatesTable};\n\n" +
                "$verified_hosts_without_important_urls = SELECT host_id, host_url  FROM ${verifiedHostsTable} WHERE " +
                "visible='true' AND (DateTime::ToSeconds(CurrentUtcTimestamp(1)) - DateTime::ToSeconds" +
                "(DateTime::MakeTimestamp(DateTime::ParseIso8601(verification_date)))) > " +
                "$secondsBetweenVerificationAndAutoAdd " +
                "AND host_id NOT IN $hosts_with_important_urls AND host_url NOT IN $important_urls_updates GROUP BY " +
                "host_id, host_url;\n\n" +
                "$main_mirrors_without_important_urls = SELECT a.host_id AS host_id, a.host_url AS host_url FROM " +
                "$verified_hosts_without_important_urls AS a LEFT JOIN " +
                "${mirrosPath}  AS b ON a.host_url=b.`Host` WHERE b.`Host` IS NULL OR b.`Host`==b.`MainHost`;\n\n" +
                "$historyFor90daysJoinWithTopUrls = SELECT Host, Path, cnt FROM $historyFor90days AS a INNER JOIN " +
                "(SELECT key, path FROM ${topUrlsTable} " +
                "WHERE path != '/' AND clicks > 5 ) AS b ON a.Host=b.key AND a.Path=b.path;\n\n" +
                "$top9_urls = SELECT Host AS host_url, TOP_BY(Path, cnt, 25) AS paths FROM " +
                "$historyFor90daysJoinWithTopUrls GROUP BY Host;\n\n" +
                "$morda = SELECT key AS host_url, '/' AS morda FROM ${topUrlsTable} WHERE path == '/' AND clicks > 5 " +
                "GROUP BY key;\n\n" +
                "$top_urls = SELECT host_url, paths FROM ( " +
                "SELECT if(a.host_url is not null, a.host_url, b.host_url) AS host_url, ListExtend(IF(b.morda IS NOT " +
                "NULL, asList('/'), asList()), IF(a.paths IS NOT NULL, CAST(a.paths AS List<String>), asList())) as " +
                "paths\n" +
                " FROM $top9_urls AS a FULL JOIN $morda AS b USING(host_url)) WHERE paths is not null and ListLength" +
                "(paths) > 0;\n\n" +
                "INSERT INTO ${tmpTable} WITH TRUNCATE " +
                "SELECT a.host_id AS hostId, b.paths AS paths FROM $main_mirrors_without_important_urls AS a " +
                "INNER JOIN $top_urls AS b USING (host_url)";

        String query = sub.replace(template);

        YqlQueryBuilder qb = new YqlQueryBuilder();
        qb.inferSchema(YqlQueryBuilder.InferSchemaMode.INFER);
        return qb.appendText(query).build();
    }

    @Override
    public Result run(UUID runId) throws Exception {
        state = new TaskState();
        setState(state);
        DateTime now = DateTime.now();
        String yqlQuery = createYqlQuery();
        yqlService.execute(yqlQuery);
        Batch batch = new Batch();

        ytService.withoutTransaction(x -> {
            var tableReader = new AsyncTableReader<>(x, tmpTable, YtTableRange.all(),
                    YtTableReadDriver.createYSONDriver(Row.class));

            try (var reader = tableReader.read()) {
                while (reader.hasNext()) {
                    Row next = reader.next();
                    WebmasterHostId hostId = IdUtils.stringToHostId(next.getHostId());

                    if (fastSpamHostFilter.checkHost(hostId)) {
                        state.hostsWithBansSkip++;
                        continue;
                    }

                    Set<ImportantUrlRequest> urls = next.getPaths().stream().map(
                            path -> new ImportantUrlRequest(
                                    hostId,
                                    path,
                                    now)
                    ).collect(Collectors.toSet());

                    batch.add(urls, hostId, next.getPaths());
                    if (batch.size() > 3000) {
                        batch.process(state);
                        batch.clear();
                    }
                }
                if (batch.size() > 0) {
                    batch.process(state);
                    batch.clear();
                }
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
            return true;
        });
        return new Result(TaskResult.SUCCESS);
    }

    @lombok.Value
    public static class Row {
        String hostId;
        List<String> paths;
    }



    @Getter
    private class Batch {
        Set<ImportantUrlRequest> urls = new HashSet<>();
        List<WMCEventContent> events = new ArrayList<>();

        public void add(Set<ImportantUrlRequest> urls, WebmasterHostId hostId, List<String> paths) {
            this.urls.addAll(urls);
            this.events.add(makeEvent(hostId, paths));
        }

        public void process(TaskState state) {
            importantUrlsYDao.add(urls);
            hostsWithImportantUrlsYDao.add(urls);
            wmcEventsService.addEventContents(events);
            state.addedHostsWithoutImportantUrls += events.size();
        }

        public void clear() {
            urls.clear();
            events.clear();
        }

        // так как urls.size() >= events.size()
        public int size() {
            return urls.size();
        }

        private WMCEventContent makeEvent(WebmasterHostId hostId, List<String> urls) {

            return new RetranslateToUsersEvent<>(
                    UserHostMessageEvent.create(
                            hostId,
                            new MessageContent.AutoAddedImportantUrls(hostId, urls),
                            NotificationType.IMPORTANT_URLS_AUTO_ADD,
                            false)
            );
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORTANT_URLS_AUTO_ADD;
    }

    @Override
    //будни в 7-8 утра
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 7 * * MON-FRI");
    }

    @Getter
    public static class TaskState implements PeriodicTaskState {
        private long addedHostsWithoutImportantUrls = 0;
        private long hostsWithBansSkip = 0;
    }
}
