package ru.yandex.webmaster3.worker.importanturls;

import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import ru.yandex.webmaster3.core.codes.HttpCodeGroup;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlStatus;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlStatus.SearchInfo;
import ru.yandex.webmaster3.tanker.I18nHttpCodeGroup;
import ru.yandex.webmaster3.tanker.I18nHttpCodeStatusDescription;
import ru.yandex.webmaster3.tanker.I18nHttpCodeStatusTitle;

/**
 * @author Oleg Bazdyrev
 */
public class I18nUtils {
    private static final DateTimeFormatter DATE = DateTimeFormat.forPattern("dd.MM.yyyy");

    /**
     * Возвращает локализованное название статуса страницы в поиске
     *
     * @param status
     * @param language
     * @return
     */
    public static String getUrlStatusTitle(SearchInfo status, LanguageEnum language) {
        if (status == null || status.getUrlSearchInfo() == null || status.getUrlSearchInfo().getStatus() == null) {
            return null;
        }
        StringBuilder sb = new StringBuilder(128);
        I18nHttpCodeStatusTitle.fromEnum(status.getUrlSearchInfo().getStatus())
                .newBuilder(language)
                .httpCode(null)
                .renderTo(sb);
        return sb.toString();
    }

    /**
     * Возвращет локализованное название HTTP-кода
     *
     * @param httpCode
     * @param language
     * @return
     */
    public static String getHttpCodeGroupName(HttpCodeInfo httpCode, LanguageEnum language) {
        if (httpCode == null) {
            return null;
        }
        HttpCodeGroup httpCodeGroup = httpCode.getHttpCodeGroup();
        if (httpCodeGroup.ordinal() >= HttpCodeGroup.HTTP_200_OK.ordinal() && httpCodeGroup.ordinal() <
                HttpCodeGroup.HTTP_GROUP_2XX.ordinal()) {
            // возвращаем обычное число
            return String.valueOf(httpCodeGroup.value());
        }
        try {
            return I18nHttpCodeGroup.fromEnum(httpCodeGroup).getText(language);
        } catch (IllegalArgumentException e) {
            // возвращаем обычное число, если такого кода нет в танкере
            return String.valueOf(httpCodeGroup.value());
        }
    }

    public static String extendedDescription(
            WebmasterHostId hostId,
            String path,
            ImportantUrlStatus.SearchInfo searchInfo,
            LanguageEnum lang) {
        SearchUrlStatusEnum status = searchInfo.getUrlSearchInfo().getStatus();
        String encodedOriginalUrl = URLEncodeUtil.urlEncode(IdUtils.hostIdToUrl(hostId) + path);
        return I18nHttpCodeStatusDescription.fromEnum(status).newBuilder(lang)
                .beautyUrl(searchInfo.getUrlSearchInfo().getBeautyUrl())
                .encodedOriginalUrl(encodedOriginalUrl)
                .hostId(hostId.toStringId())
                .httpCode(searchInfo.getHttpCode().getCode().toString())
                .lastAccess(searchInfo.getLastAccess().toString(DATE))
                .redirectTarget(searchInfo.getUrlSearchInfo().getRedirectTarget())
                .relCanonicalTarget(searchInfo.getUrlSearchInfo().getRelCanonicalTarget())
                .render();
    }
}
