package ru.yandex.webmaster3.worker.importanturls;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.datastax.driver.core.utils.UUIDs;
import com.google.common.collect.ImmutableMap;
import lombok.Setter;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.importanturls.dao.ImportantUrlsChangesCHDao;
import ru.yandex.webmaster3.storage.notifications.NotificationProgress;
import ru.yandex.webmaster3.storage.notifications.dao.NotificationProgressCypressDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.turbo.AbstractYqlPrepareImportTask;

import static ru.yandex.webmaster3.storage.importanturls.dao.ImportantUrlsChangesCHDao.F;
import static ru.yandex.webmaster3.storage.importanturls.dao.ImportantUrlsChangesCHDao.TABLE;
import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState.DONE;

/**
 * Created by Oleg Bazdyrev on 2019-05-28.
 */
public class ImportImportantUrlsChangesTask extends AbstractYqlPrepareImportTask {

    private static final int LINES_COUNT = 1024;
    private static final Map<String, String> SOURCE_EXPRESSIONS = new ImmutableMap.Builder<String, String>()
            .put(F.DATE, "'0000-00-00'")
            .put(F.HOST_ID, "host_id")
            .put(F.NOTIFICATION_TYPE, "notification_type")
            .put(F.USER_ID, "cast(user_id as String)")
            .put(F.LOGIN, "nvl(login, '')")
            .put(F.FIO, "nvl(fio, '')")
            .put(F.EMAIL, "nvl(email, '')")
            .put(F.LANGUAGE, "nvl(language, 'RU')")
            .put(F.CHANNEL_EMAIL, "if(channel_email, '1', '0')")
            .put(F.CHANNEL_SERVICE, "if(channel_service, '1', '0')")
            .put(F.CHANGES, "String::ReplaceAll(cast(Yson::SerializeJson(Changes) as String), @@\\@@, @@\\\\@@)")
            .build();

    private static final long TABLE_AGE_FOR_REMOVE = TimeUnit.DAYS.toMillis(10);
    @Setter
    private ImportantUrlsChangesCHDao importantUrlsChangesCHDao;
    @Setter
    private NotificationProgressCypressDao importantUrlsNotificationProgressCypressDao;
    @Setter
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;
    @Setter
    private boolean enableNotifications;

    @Override
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad imprt) throws Exception {
        return ytService.withoutTransactionQuery(cypressService -> {
            List<YtPath> tables = cypressService.list(tablePath);
            long lastTimestamp = Optional.ofNullable(imprt.getData()).map(Long::valueOf).orElse(0L);
            // ищем первую необработанную таблицу
            YtPath table = tables.stream().filter(path -> Long.parseLong(path.getName()) > lastTimestamp)
                    .min(Comparator.naturalOrder()).orElse(null);
            if (table != null) {
                long newTimestamp = Long.parseLong(table.getName());
                LocalDate date = new DateTime(newTimestamp).toLocalDate();
                return imprt.withData(table.getName()).withSourceTable(table, date, date);
            }
            // ничего не нашли, сразу завершимся
            log.info("Expected table not found, nothing to import");
            return imprt.withState(DONE);
        });
    }

    @Override
    protected int getShardsCount() {
        return 1; // всегда будем лить на первый шард
    }

    @Override
    protected CHTable getTable() {
        return ImportantUrlsChangesCHDao.TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.IMPORTANT_URLS_CHANGES;
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        String fields = getTable().getFields().stream().map(CHField::getName).map(SOURCE_EXPRESSIONS::get)
                .collect(Collectors.joining(" || '\\t' || ", "(", " || '\\n')"));

        YqlQueryBuilder yqlQueryBuilder = YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendFDefinition(YqlFunctions.URL_2_HOST_ID)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM (\n")
                .appendText("SELECT 0 as ShardId,")
                .appendText("(Digest::Fnv64(host_id) % " + LINES_COUNT + ") as RowId,")
                .appendText(fields).appendText("as data ")
                .appendText("FROM")
                .appendTable(imprt.getSourceTable())
                .appendText(") \n GROUP BY ShardId, RowId;")
                .appendText("COMMIT;\n\n");

        return yqlQueryBuilder;
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        long timestamp = Long.parseLong(imprt.getData());
        UUID notificationId = UUIDs.startOf(timestamp);

        if (enableNotifications) {
            if (!importantUrlsNotificationProgressCypressDao.haveProgressRecord(notificationId)) {
                int targetsCount = importantUrlsChangesCHDao.countRecords(String.valueOf(timestamp));
                log.info("Important urls changes notifications {}:{} contains {} targets to send",
                        notificationId, timestamp, targetsCount);
                int chunkSize = Math.min(targetsCount, 1000);
                importantUrlsNotificationProgressCypressDao.createRecord(
                        notificationId,
                        NotificationProgress.createInitial(chunkSize, targetsCount, 0)
                );
                log.info("ZK record created for new domains notification {}:{}", notificationId, timestamp);
            }
        } else {
            log.info("Notifications disabled");
        }
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad clean(YtClickhouseDataLoad imprt) throws Exception {
        ytService.withoutTransactionQuery(cypressService -> {
            List<YtPath> tables = cypressService.list(tablePath);
            long timestampToDelete = Long.parseLong(imprt.getData()) - TABLE_AGE_FOR_REMOVE;
            // ищем первую необработанную таблицу
            tables.stream().filter(path -> Long.parseLong(path.getName()) < timestampToDelete)
                    .forEach(cypressService::remove);
            return true;
        });
        // чистка в Clickhouse
        List<String> changesTables = clickhouseSystemTablesCHDao.getTableNamesForPrefix(
                TABLE.getDatabase(), TABLE.replicatedMergeTreeTableName(-1, ""), getShardsCount())
                .stream().sorted(Comparator.reverseOrder()).skip(30).collect(Collectors.toList());
        // dropping old tables
        for (String changesTable : changesTables) {
            String query = "DROP TABLE IF EXISTS " + TABLE.getDatabase() + "." + changesTable;
            clickhouseServer.executeOnAllHosts(ClickhouseQueryContext.useDefaults(), query, getShardsCount());
        }

        // чистка изменений на YT и в Clickhouse
        return super.clean(imprt);
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.IMPORTANT_URLS_CHANGES;
    }

}
