package ru.yandex.webmaster3.worker.links;

import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.collect.Lists;
import lombok.Setter;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.dao.ClickhouseTablesRepository;
import ru.yandex.webmaster3.storage.links.LinksImportService;
import ru.yandex.webmaster3.storage.searchquery.SearchQueriesConstants;
import ru.yandex.webmaster3.storage.searchquery.importing.dao.YtClickhouseDataLoadRepository;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.ytimport.AbstractSwitchDatesTask;
import ru.yandex.webmaster3.worker.ytimport.AbstractYtClickhouseDataLoadTask;

import static ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao.DB_WEBMASTER3_LINKS;

/**
 * Created by Oleg Bazdyrev on 12/04/2017.
 */
public class LinksSwitchDatesTask extends AbstractSwitchDatesTask<PeriodicTaskState> {

    private static final Logger log = LoggerFactory.getLogger(AbstractYtClickhouseDataLoadTask.class);
    private static final Pattern TABLE_NAME_PATTERN = Pattern.compile(DB_WEBMASTER3_LINKS +
            ".(int|ext|ext_gone)_(" + SearchQueriesConstants.IN_TABLE_NAME_DATE_PATTERN + ")_distrib");

    private static final String LINK_STATISTICS_NAME_PREFIX = "tmp_link_statistics_part_";
    private static final Pattern LINK_STATISTICS_NAME_PATTERN = Pattern.compile(
            LINK_STATISTICS_NAME_PREFIX + "(" + SearchQueriesConstants.IN_TABLE_NAME_DATE_PATTERN + ")");

    private static final Set<YtClickhouseDataLoadType> LOAD_TYPES = EnumSet.copyOf(Lists.newArrayList(
            YtClickhouseDataLoadType.INTERNAL_LINKS,
            YtClickhouseDataLoadType.EXTERNAL_LINKS,
            YtClickhouseDataLoadType.EXTERNAL_GONE_LINKS,
            YtClickhouseDataLoadType.LINK_STATISTICS));

    private static final Set<TableType> TABLE_TYPES = EnumSet.copyOf(Lists.newArrayList(
            TableType.INTERNAL_LINK_SAMPLES,
            TableType.EXTERNAL_LINK_SAMPLES,
            TableType.EXTERNAL_DELETED_LINK_SAMPLES
    ));

    @Setter
    private YtClickhouseDataLoadRepository ytClickhouseDataLoadCDao;
    @Setter
    private ClickhouseTablesRepository clickhouseTablesCDao;
    @Setter
    private String schedule;
    @Setter
    private PeriodicTaskType type;

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Checking dates for links tables");
        List<YtClickhouseDataLoad> imports = ytClickhouseDataLoadCDao.listAll();
        // найдем минимальную дату, по которой есть все обновления
        LocalDate newMaximumDate = imports.stream().filter(sqi -> LOAD_TYPES.contains(sqi.getType()))
                .map(YtClickhouseDataLoad::getMaxProcessedDate).min(Comparator.naturalOrder()).orElse(null);
        log.info("New maximum date for links tables: {}", newMaximumDate);
        // переименовываем таблицы статистики
        renameTables(newMaximumDate, DB_WEBMASTER3_LINKS, LINK_STATISTICS_NAME_PREFIX, LINK_STATISTICS_NAME_PATTERN,
                clickhouseServer.getShardsCount());
        // переводим в "онлайн" самые последние таблицы в webmaster3_internal.clickhouse_tables
        switchOnLinksTables(newMaximumDate);
        // обновляем диапазон дат
        return new Result(TaskResult.SUCCESS);
    }

    protected void switchOnLinksTables(LocalDate newMaximumDate) throws Exception {
        for (ClickhouseTableInfo table : clickhouseTablesCDao.listTables()) {
            // нужны только TABLE_TYPES
            if (TABLE_TYPES.contains(table.getType())) {
                if (table.getState() == TableState.ON_LINE) {
                    continue;
                }
                Matcher matcher = TABLE_NAME_PATTERN.matcher(table.getClickhouseFullName());
                if (!matcher.matches()) // может быть какая-то старая таблица
                    continue;
                // проверим дату
                LocalDate tableDate = SearchQueriesConstants.IN_TABLE_NAME_DATE_FORMATTER.parseLocalDate(matcher.group(2));
                if (!tableDate.isAfter(newMaximumDate)) {
                    // не очень красиво, отрезаем имя бд и суффикс _distrib
                    String prefix = table.getClickhouseFullName();
                    prefix = prefix.substring(prefix.indexOf('.') + 1, prefix.length() - "_distrib".length());
                    prefix += LinksImportService.INTERNAL_LINKS_TABLE.getPartNameSuffix();
                    // проверим, что все таблички непустые
                    clickhouseSystemPartsCHDao.checkTablesNotEmpty(DB_WEBMASTER3_LINKS, prefix);
                    clickhouseTablesCDao.update(table.withState(TableState.ON_LINE));
                }
            }
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return type;
    }

    @Override
    public TaskSchedule getSchedule() {
        return schedule == null ? TaskSchedule.never() : TaskSchedule.startByCron(schedule);
    }

}
