package ru.yandex.webmaster3.worker.memorandum;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableSource;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.dao.ClickhouseTablesRepository;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationCommand;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationPriority;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtMapReduceCommand;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtOperationId;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtUtils;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.storage.ytimport.ImportPriority;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportCommand;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseTableRelation;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.ytimport.AbstractYtClickhouseDataLoadTask;

/**
 * @author avhaliullin
 */
public class ImportMemorandumInfoTask extends AbstractYtClickhouseDataLoadTask {
    private static final int SAMPLES_COUNT_LIMIT = 50_000;
    private static final String TASK_NAME = "MEMORANDUM";
    private static final DateTimeFormatter CH_TABLE_DT_FORMATTER = DateTimeFormat.forPattern("yyyyMMdd'T'HHmmss");
    private static final Duration RECORDS_KEEP_DURATION = Duration.standardDays(3L); // храним записи за последние 3 дня

    @Setter
    private YqlService yqlService;
    @Setter
    private CommonDataStateYDao commonDataStateYDao;
    @Setter
    private ClickhouseTablesRepository clickhouseTablesCDao;
    @Setter
    private YtPath destinationDirPath;
    @Setter
    private YtPath ytWorkDir;
    @Setter
    private YtPath ytPrepareExecutable;
    @Setter
    private CommonDataType dataType;

    private static final CHTable MEMORANDUM_CH_TABLE = CHTable.builder()
            .database(AbstractClickhouseDao.DB_WEBMASTER3_SEARCHURLS)
            .name("memorandum_url_samples_%s")
            .partitionBy("cityHash64(host_id) % 16")
            .keyField("host_id", CHPrimitiveType.String)
            .keyField("path", CHPrimitiveType.String)
            .field("copyright_object", CHPrimitiveType.String)
            .field("copyright_owner", CHPrimitiveType.String)
            .build();

    @Override
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad latestImport) throws Exception {
        return ytService.inTransaction(tablePath).query(cypressService -> {
            NavigableMap<Instant, Map<MemorandumYtTableType, YtPath>> destTables = Collections.emptyNavigableMap();
            if (cypressService.exists(destinationDirPath)) {
                List<YtPath> destTablesList = cypressService.list(destinationDirPath);
                destTables = MemorandumYTUtils.parseMemorandumTableNames(destTablesList);
            }
            Instant lastPreparedTableDate = destTables.isEmpty() ? null : destTables.lastKey();
            Instant sourceLastUpdate = cypressService.getNode(tablePath).getUpdateTime().toInstant();
            log.info("Last prepared table date {}, last source update {}", lastPreparedTableDate, sourceLastUpdate);
            if (lastPreparedTableDate == null || lastPreparedTableDate.isBefore(sourceLastUpdate)) {
                makeAllUrlsTable(cypressService, sourceLastUpdate);
                makeStatsTable(cypressService, sourceLastUpdate);
                makeSamplesTable(cypressService, sourceLastUpdate);
                importMemorandumStats(cypressService, sourceLastUpdate);
                LocalDate importDate = sourceLastUpdate.toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE).toLocalDate();
                return latestImport.withSourceTable(
                        makeTablePath(sourceLastUpdate, MemorandumYtTableType.SAMPLES),
                        importDate, importDate
                ).withData(String.valueOf(sourceLastUpdate.getMillis()));
            } else {
                log.info("Nothing to update");
                return latestImport.withState(YtClickhouseDataLoadState.DONE);
            }
        });
    }

    protected void importMemorandumStats(YtCypressService cypressService, Instant tableDate) {

    }

    @Override
    protected YtClickhouseDataLoad prepare(YtClickhouseDataLoad imprt) throws Exception {
        List<YtPath> tables = preparedTables(imprt);
        ytService.inTransaction(ytPrepareExecutable).execute(cypressService -> {
            YtUtils.recreateTables(cypressService, tables, new YtNodeAttributes().setCompressionCodec("none"));
            YtMapReduceCommand mrCommand = YtUtils.newPrepareTablesForImportBuilder()
                    .addInputTable(imprt.getSourceTable())
                    .setOutputTables(tables)
                    .setTask(TASK_NAME)
                    .setBinary(ytPrepareExecutable)
                    .buildV2();

            YtOperationId operationId = cypressService.mapReduce(mrCommand);
            if (!cypressService.waitFor(operationId)) {
                throw new YtException("Prepare failed. See " + operationId);
            }
            return true;
        });
        return imprt.withPreparedTables(tables).withNextState();
    }

    @Override
    protected YtClickhouseDataLoad doImport(YtClickhouseDataLoad imprt) throws Exception {
        String dateString = dateForCHTableName(imprt);
        List<YtPath> tables = imprt.getPreparedTables();
        int shards = tables.size();
        UUID taskId = UUIDs.timeBased();
        List<YtClickhouseTableRelation> tablesRels = new ArrayList<>();
        int idx = 0;
        for (int shard = 0; shard < shards; shard++) {
            tablesRels.add(new YtClickhouseTableRelation(
                    tables.get(idx++),
                    shard,
                    MEMORANDUM_CH_TABLE.replicatedMergeTreeTableName(-1, dateString),
                    MEMORANDUM_CH_TABLE.createReplicatedMergeTreeSpec(-1, dateString)
            ));
        }
        YtClickhouseImportCommand command = new YtClickhouseImportCommand(
                taskId,
                tablesRels,
                MEMORANDUM_CH_TABLE.getDatabase(),
                MEMORANDUM_CH_TABLE.importSpec(),
                ImportPriority.ONLINE
        );
        ytClickhouseImportManager.startImport(command);
        return imprt.withImportTaskIds(taskId)
                .withNextState();
    }

    @Override
    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        String dateString = dateForCHTableName(imprt);
        UUID taskId = UUIDs.timeBased();
        log.info("Replication id " + taskId);
        List<ClickhouseReplicationCommand.TableInfo> tables = new ArrayList<>();
        int shardsCount = imprt.getPreparedTables().size();
        for (int shard = 0; shard < shardsCount; shard++) {
            tables.add(new ClickhouseReplicationCommand.TableInfo(
                    MEMORANDUM_CH_TABLE.replicatedMergeTreeTableName(-1, dateString),
                    MEMORANDUM_CH_TABLE.createReplicatedMergeTreeSpec(-1, dateString),
                    shard
            ));
        }
        ClickhouseReplicationCommand command = new ClickhouseReplicationCommand(
                taskId,
                MEMORANDUM_CH_TABLE.getDatabase(),
                ClickhouseReplicationPriority.ONLINE,
                tables
        );
        clickhouseReplicationManager.enqueueReplication(command);
        return imprt.withReplicationTaskIds(taskId)
                .withNextState();
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        // надо бы переименовать этот шаг во что-нибудь более общее. Deploy, activate и т.д.
        commonDataStateYDao.update(new CommonDataState(
                dataType,
                String.valueOf(parseTableTime(imprt).getMillis()),
                DateTime.now()
        ));
        String chTableName = MEMORANDUM_CH_TABLE.getDatabase() + "." + MEMORANDUM_CH_TABLE.replicatedMergeTreeTableName(0, dateForCHTableName(imprt));
        clickhouseTablesCDao.update(
                new ClickhouseTableInfo(TableType.MEMORANDUM_SAMPLES, UUIDs.timeBased(),
                        TableState.ON_LINE, DateTime.now(), TableSource.YT_HAHN, null,
                        chTableName, chTableName, chTableName, imprt.getPreparedTables().size(), 1)
        );
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad clean(YtClickhouseDataLoad imprt) throws Exception {
        // почистим destinationDirPath от старья
        ytService.inTransaction(destinationDirPath).execute(cypressService -> {
            if (cypressService.exists(destinationDirPath)) {
                List<YtPath> destTablesList = cypressService.list(destinationDirPath);
                NavigableMap<Instant, Map<MemorandumYtTableType, YtPath>> destTables =
                        MemorandumYTUtils.parseMemorandumTableNames(destTablesList);
                destTables.headMap(Instant.now().minus(RECORDS_KEEP_DURATION))
                        .forEach((instant, map) -> map.values().forEach(cypressService::remove));
            }
            return true;
        });

        return super.clean(imprt);
    }

    private void makeAllUrlsTable(YtCypressService cypressService, Instant tableDate) {
        YtPath allUrlsTable = makeTablePath(tableDate, MemorandumYtTableType.ALL_URLS);
        log.info("Preparing all urls table {}", allUrlsTable);
        yqlService.execute(
                YqlQueryBuilder.newBuilder()
                        .inferSchema(YqlQueryBuilder.InferSchemaMode.INFER)
                        .cluster(tablePath)
                        .transaction(cypressService.getTransactionId())
                        .appendText("PRAGMA dq.AnalyzeQuery=\"0\";\n")
                        .appendText("INSERT INTO")
                        .appendTable(allUrlsTable)
                        .appendText("SELECT HostId, Path, CopyrightObject, CopyrightOwner")
                        .appendText("FROM")
                        .appendTable(tablePath)
                        .appendText("WHERE ")
                        .appendFCall(YqlFunctions.url2HostId("Url::NormalizeWithDefaultHttpScheme(Url)")).appendText(" IS NOT NULL ")
                        .appendText("GROUP BY ").appendFCall(YqlFunctions.url2HostId("Url::NormalizeWithDefaultHttpScheme(Url)")).appendText("AS HostId,")
                        .appendText(" Url::GetTail(Url) as Path, CopyrightObject, CopyrightOwner")
                        .appendText("ORDER BY HostId, Path")
                        .appendText(";")
                        .build()
        );
    }

    private void makeStatsTable(YtCypressService cypressService, Instant tableDate) {
        YtPath allUrlsTable = makeTablePath(tableDate, MemorandumYtTableType.ALL_URLS);
        YtPath statsTable = makeTablePath(tableDate, MemorandumYtTableType.STATS);
        log.info("Preparing stats table {}", statsTable);
        yqlService.execute(
                YqlQueryBuilder.newBuilder()
                        .cluster(allUrlsTable)
                        .transaction(cypressService.getTransactionId())
                        .appendText("PRAGMA dq.AnalyzeQuery=\"0\";\n")
                        .appendText("INSERT INTO")
                        .appendTable(statsTable)
                        .appendText("SELECT HostId, count(*) AS Count")
                        .appendText("FROM")
                        .appendTable(allUrlsTable)
                        .appendText("GROUP BY HostId")
                        .appendText("ORDER BY HostId")
                        .appendText(";")
                        .build()
        );
    }

    private void makeSamplesTable(YtCypressService cypressService, Instant tableDate) {
        //TODO: Optimize using https://ml.yandex-team.ru/thread/yql/167477611142853676/
        YtPath allUrlsTable = makeTablePath(tableDate, MemorandumYtTableType.ALL_URLS);
        YtPath samplesTable = makeTablePath(tableDate, MemorandumYtTableType.SAMPLES);
        log.info("Preparing samples table {}", samplesTable);
        yqlService.execute(
                YqlQueryBuilder.newBuilder()
                        .cluster(allUrlsTable)
                        .transaction(cypressService.getTransactionId())
                        .appendText("PRAGMA dq.AnalyzeQuery=\"0\";\n")
                        .appendText("PRAGMA yt.MaxRowWeight = \"128M\";")
                        .appendText("INSERT INTO")
                        .appendTable(samplesTable)
                        .appendText("SELECT")
                        .appendText("    HostId,")
                        .appendText("    Row.Path AS Path,")
                        .appendText("    Row.CopyrightObject AS CopyrightObject,")
                        .appendText("    Row.CopyrightOwner AS CopyrightOwner")
                        .appendText("FROM (")
                        .appendText("    SELECT")
                        .appendText("        HostId,")
                        .appendText("        Unwrap(TOP_BY(AsStruct(")
                        .appendText("            Path AS Path,")
                        .appendText("            CopyrightObject AS CopyrightObject,")
                        .appendText("            CopyrightOwner AS CopyrightOwner")
                        .appendText("        ), Path, " + SAMPLES_COUNT_LIMIT + ")) AS Row ")
                        .appendText("    FROM")
                        .appendTable(allUrlsTable)
                        .appendText("    GROUP BY HostId")
                        .appendText(")")
                        .appendText("FLATTEN BY Row")
                        .appendText("ORDER BY HostId, Path;")
                        .build()
        );
    }

    protected YtPath makeTablePath(Instant tableDate, MemorandumYtTableType type) {
        return YtPath.path(
                destinationDirPath,
                MemorandumYTUtils.makeTableName(tableDate, type)
        );
    }

    private static String dateForCHTableName(YtClickhouseDataLoad imprt) {
        Instant tableTime = parseTableTime(imprt);
        return tableTime.toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE).toString(CH_TABLE_DT_FORMATTER);
    }

    private static Instant parseTableTime(YtClickhouseDataLoad imprt) {
        if (imprt != null && !StringUtils.isEmpty(imprt.getData())) {
            return new Instant(Long.parseLong(imprt.getData()));
        } else {
            return null;
        }
    }

    private List<YtPath> preparedTables(YtClickhouseDataLoad importData) {
        List<YtPath> tables = new ArrayList<>();
        final String dateString = String.valueOf(parseTableTime(importData).getMillis());
        for (int shard = 0; shard < clickhouseServer.getShardsCount(); shard++) {
            tables.add(YtPath.path(ytWorkDir, dateString + "_shard_" + shard));
        }
        return tables;
    }

    @Override
    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        String dateString = dateForCHTableName(imprt);
        MEMORANDUM_CH_TABLE.updateDistributedSymlink(clickhouseServer, dateString);
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.MEMORANDUM;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_MEMORANDUM_DATA;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }

}
