package ru.yandex.webmaster3.worker.metrika;

import java.util.Optional;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.log.EventLogService;
import ru.yandex.webmaster3.core.log.LogfellerEvent;
import ru.yandex.webmaster3.core.metrika.counters.CounterBindingStateEnum;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.MetrikaCounterBindingStateChangeTaskData;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskType;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.worker.Task;

/**
 * @author leonidrom
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Service
public class MetrikaCounterBindingStateChangeTask extends Task<MetrikaCounterBindingStateChangeTaskData> {
    private final SiteProblemsService siteProblemsService;
    private final EventLogService eventLogService;

    @Override
    public Result run(MetrikaCounterBindingStateChangeTaskData stateChangeEvent) throws Exception {
        WebmasterHostId domainHostId = IdUtils.urlToHostId(stateChangeEvent.getDomain());
        log.info("Got event for: {}, state: {}", domainHostId, stateChangeEvent.getState());
        if (stateChangeEvent.getState() == CounterBindingStateEnum.APPROVED) {
            // быстро выключим пользователю проблему, не дожидаясь запуска периодической таски
            var problemSignal = new ProblemSignal(SiteProblemTypeEnum.NO_METRIKA_COUNTER_BINDING, SiteProblemState.ABSENT, stateChangeEvent.getEventTime());
            siteProblemsService.updateCleanableProblem(domainHostId, problemSignal);
        }
        eventLogService.checkedSend(new LogfellerEvent(WorkerTaskType.METRIKA_COUNTER_BINDING_CHANGE.name(),
                stateChangeEvent.getTaskId(),
                stateChangeEvent,
                Optional.ofNullable(stateChangeEvent.getEventTime()).orElse(DateTime.now()),
                MetrikaCounterBindingStateChangeTask.class.getSimpleName(),
                DateTime.now()));
        return Result.SUCCESS;
    }

    @Override
    public Class<MetrikaCounterBindingStateChangeTaskData> getDataClass() {
        return MetrikaCounterBindingStateChangeTaskData.class;
    }
}
