package ru.yandex.webmaster3.worker.metrika;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.domain.HostDomainInfoService;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCountersUtil;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.services.ServiceInfo;
import ru.yandex.webmaster3.storage.services.SiteServiceType;
import ru.yandex.webmaster3.storage.services.SiteServicesCHDao;
import ru.yandex.webmaster3.storage.services.UseMetrikaInfo;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author leonidrom
 */
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Slf4j
public class UpdateMetrikaBindingProblemsPeriodicTask extends PeriodicTask<UpdateMetrikaBindingProblemsPeriodicTask.TaskState> {
    private static String[] SITE_SERVICES_COLUMNS = {SiteServicesCHDao.F.USE_METRIKA, SiteServicesCHDao.F.OWNER};
    private final static int TOTAL_THREADS = 32;
    private final static int BATCH_SIZE = 2048;

    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;
    private final SiteServicesCHDao mdbSiteServicesCHDao;
    private final SiteProblemsService siteProblemsService;
    private final MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private final HostOwnerService hostOwnerService;

    private Set<String> domainsWithApprovedBindings = new HashSet<>();

    @Override
    public Result run(UUID runId) throws Exception {
        var updateStartTime = DateTime.now();
        setState(new UpdateMetrikaBindingProblemsPeriodicTask.TaskState());

        domainsWithApprovedBindings = collectDomainsWithApprovedBindings();
        var executorService = ru.yandex.common.util.concurrent.Executors.newBlockingFixedThreadPool(
                TOTAL_THREADS, TOTAL_THREADS,
                0, TimeUnit.MILLISECONDS,
                new ArrayBlockingQueue<>(TOTAL_THREADS),
                Executors.defaultThreadFactory());

        List<Future<TaskState>> futures = new ArrayList<>();
        List<WebmasterHostId> batch = new ArrayList<>();
        allVerifiedHostsCacheService.foreachHost(hostId -> {
            if (HostDomainInfoService.isSubdomain(hostId) && hostOwnerService.getHostOwner(hostId).equals(hostId)) {
                // на таких поддоменах у пользователя может не быть доступа к счетчику
                return;
            }

            batch.add(hostId);
            if (batch.size() >= BATCH_SIZE) {
                var batchCopy = new ArrayList<>(batch);
                futures.add(executorService.submit(() -> processBatch(batchCopy)));
                batch.clear();
            }
        });

        if (!batch.isEmpty()) {
            futures.add(executorService.submit(() -> processBatch(new ArrayList<>(batch))));
        }

        for (var f : futures) {
            state.accumulate(f.get());
        }

        executorService.shutdownNow();
        siteProblemsService.notifyCleanableProblemUpdateFinished(SiteProblemTypeEnum.NO_METRIKA_COUNTER_BINDING,
                updateStartTime);

        return Result.SUCCESS;
    }

    private TaskState processBatch(Collection<WebmasterHostId> hostIds) throws InterruptedException {
        TaskState ts = new TaskState();

        List<String> domainsWithMetrikaInstalled = filterHostsWithMetrikaInstalled(hostIds);
        Set<String> domainsWithBinding = domainsWithMetrikaInstalled.stream()
                .filter(domain -> domainsWithApprovedBindings.contains(domain))
                .collect(Collectors.toSet());
        Map<WebmasterHostId, ProblemSignal> map = new HashMap<>(BATCH_SIZE);
        for (String domain : domainsWithMetrikaInstalled) {
            var domainHostId = IdUtils.urlToHostId(domain);
            ProblemSignal problemSignal;
            if (!domainsWithBinding.contains(domain)) {
                problemSignal = new ProblemSignal(new SiteProblemContent.NoMetrikaCounterBinding(), DateTime.now());
                map.put(domainHostId, problemSignal);
            }
        }
        siteProblemsService.updateCleanableProblems(map, SiteProblemTypeEnum.NO_METRIKA_COUNTER_BINDING);

        ts.domainsWithMetrikaCounter = domainsWithMetrikaInstalled.size();
        ts.domainsWithMetrikaBinding = domainsWithBinding.size();
        ts.domainsWithoutMetrikaBinding = ts.domainsWithMetrikaCounter - ts.domainsWithMetrikaBinding;

        return ts;
    }

    private List<String> filterHostsWithMetrikaInstalled(Collection<WebmasterHostId> hostIds) throws InterruptedException {

        Set<String> domains = hostIds.stream()
                .map(MetrikaCountersUtil::hostToPunycodeDomain)
                .collect(Collectors.toSet());

        Map<String, Map<SiteServiceType, ServiceInfo>> siteServices = RetryUtils.query(
                RetryUtils.instantRetry(5), () -> mdbSiteServicesCHDao.getSiteServices(domains, SITE_SERVICES_COLUMNS));

        return siteServices.entrySet().stream()
                .filter(p -> {
                    var serviceInfoMap = p.getValue();
                    boolean isMetrikaInstalled = false;
                    if (serviceInfoMap.containsKey(SiteServiceType.USE_METRIKA)) {
                        UseMetrikaInfo metrikaInfo = (UseMetrikaInfo) serviceInfoMap.get(SiteServiceType.USE_METRIKA);
                        isMetrikaInstalled = metrikaInfo.isValue();
                    }
                    return isMetrikaInstalled;
                })
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
    }

    private Set<String> collectDomainsWithApprovedBindings() {
        Set<String> res = new HashSet<>();
        metrikaCounterBindingStateYDao.forEachLink(b -> {
            if (b.getUpdateDate() == null) {
                // такого быть не должно, но на всякий
                return;
            }

            if (b.getCounterBindingState().isApproved()) {
                res.add(b.getDomain());
            }
        }, false);

        return res;
    }


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPDATE_METRIKA_BINDING_PROBLEMS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 8 * * MON");
    }

    public static class TaskState implements PeriodicTaskState {
        public int domainsWithMetrikaCounter;
        public int domainsWithoutMetrikaBinding;
        public int domainsWithMetrikaBinding;

        public void accumulate(TaskState other) {
            this.domainsWithMetrikaCounter += other.domainsWithMetrikaCounter;
            this.domainsWithoutMetrikaBinding += other.domainsWithoutMetrikaBinding;
            this.domainsWithMetrikaBinding += other.domainsWithMetrikaBinding;
        }
    }
}
