package ru.yandex.webmaster3.worker.metrika;

import com.google.common.base.Stopwatch;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrika.counters.CounterBinding;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * Created by ifilippov5 on 10.12.17.
 */
public class UploadMetrikaCounterLinksPeriodicTask extends PeriodicTask<UploadMetrikaCounterLinksPeriodicTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(UploadMetrikaCounterLinksPeriodicTask.class);

    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss");

    private static final String TABLE_SCHEMA = "[" +
            "{'name': '" + YtRow.F_COUNTER_ID + "', 'type': 'int64'}, " +
            "{'name': '" + YtRow.F_DOMAIN + "', 'type': 'string'}, " +
            "{'name': '" + YtRow.F_STATUS + "', 'type': 'string'}, " +
            "{'name': '" + YtRow.F_UPDATE_TIME + "', 'type': 'string'}, " +
            "{'name': '" + YtRow.F_WEBMASTER_UID + "', 'type': 'int64'}, " +
            "{'name': '" + YtRow.F_METRIKA_UID + "', 'type': 'int64'}, " +
            "{'name': '" + YtRow.F_ORIGIN + "', 'type': 'string'}]";

    private MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private YtPath workDir;
    private String tableName;
    private YtService ytService;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new UploadMetrikaCounterLinksPeriodicTask.TaskState());

        YtPath tablePath = YtPath.path(workDir, tableName);
        YtTableData table = null;
        try {
            table = ytService.prepareTableData(tablePath.getName(), this::uploadFromCassandra);

            Stopwatch stopwatch = Stopwatch.createStarted();
            YtTransactionService.TransactionProcess process = new YtUtils.TransactionWriterBuilder(tablePath, table)
                    .withSchema(TABLE_SCHEMA)
                    .build();

            YtUtils.TransactionExecutor writer = new YtUtils.TransactionExecutor(ytService, workDir);
            writer.execute(process);
            getState().uploadToYtMs = stopwatch.elapsed(TimeUnit.MILLISECONDS);

        } finally {
            if (table != null) {
                table.delete();
            }
        }

        return new Result(TaskResult.SUCCESS);
    }

    private void uploadFromCassandra(TableWriter tw) {
        try {
            metrikaCounterBindingStateYDao.forEachLink(link -> {
                writeToTable(tw, link);
            }, false);
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to get all counter links",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
    }

    private void writeToTable(TableWriter tw, CounterBinding link) {
        tw.column(YtRow.F_COUNTER_ID, link.getCounterId());
        tw.column(YtRow.F_DOMAIN, link.getDomain());
        tw.column(YtRow.F_STATUS, link.getCounterBindingState().getMeaning());
        tw.column(YtRow.F_UPDATE_TIME, link.getUpdateDate() == null ? "" : link.getUpdateDate().toString(DATE_FORMAT));
        tw.columnObject(YtRow.F_WEBMASTER_UID, link.getWebmasterUserId());
        tw.columnObject(YtRow.F_METRIKA_UID, link.getMetrikaUserId());
        tw.columnObject(YtRow.F_ORIGIN, link.getOrigin());

        try {
            tw.rowEnd();
        } catch (YtException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_METRIKA_COUNTER_LINKS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    @Required
    public void setMetrikaCounterBindingStateYDao(MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao) {
        this.metrikaCounterBindingStateYDao = metrikaCounterBindingStateYDao;
    }

    @Required
    public void setWorkDir(YtPath workDir) {
        this.workDir = workDir;
    }

    @Required
    public void setTableName(String tableName) {
        this.tableName = tableName;
    }

    @Required
    public void setYtService(YtService ytService) {
        this.ytService = ytService;
    }

    private static class YtRow {
        static final String F_COUNTER_ID = "counter_id";
        static final String F_DOMAIN = "domain";
        static final String F_STATUS = "status";
        static final String F_UPDATE_TIME = "update_time";
        static final String F_WEBMASTER_UID = "webmaster_uid";
        static final String F_METRIKA_UID = "metrika_uid";
        static final String F_ORIGIN = "origin";
    }

    public static class TaskState implements PeriodicTaskState {
        long uploadToYtMs;

        public long getUploadToYtMs() {
            return uploadToYtMs;
        }
    }
}
