package ru.yandex.webmaster3.worker.metrika;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.metrika.MetrikaCrawlStateService;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.metrika.data.MetrikaDomainCrawlState;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 *
 * Выгружает на Yt домен/счетчики, для которых разрешен обход.
 */
public class UploadMetrikaCrawlSettingsTask extends PeriodicTask<PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(UploadMetrikaCrawlSettingsTask.class);

    private static final YtSchema YT_TABLE_SCHEMA = new YtSchema();
    private static final YtColumn<String> DOMAIN = YT_TABLE_SCHEMA.addColumn("domain", YtColumn.Type.STRING);
    private static final YtColumn<Set<Long>> COUNTERS = YT_TABLE_SCHEMA.addColumn("counters", YtColumn.Type.any());

    private final MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private final MetrikaCrawlStateService metrikaCrawlStateService;
    private final YtService ytService;
    private YtPath exportPath;

    @Autowired
    public UploadMetrikaCrawlSettingsTask(
            MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao,
            MetrikaCrawlStateService metrikaCrawlStateService,
            YtService ytService) {
        this.metrikaCounterBindingStateYDao = metrikaCounterBindingStateYDao;
        this.metrikaCrawlStateService = metrikaCrawlStateService;
        this.ytService = ytService;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        Map<String, Set<Long>> verifiedDomainCounters = new HashMap<>();
        metrikaCounterBindingStateYDao.forEachLink(b -> {
            if (b.getCounterBindingState().isApproved()) {
                verifiedDomainCounters.computeIfAbsent(b.getDomain(), k -> new HashSet<>()).add(b.getCounterId());
            }
        }, false);

        YtTableData tableData = ytService.prepareTableData("metrika-crawl-settings", tableWriter -> {
            verifiedDomainCounters.keySet().forEach(domain -> {
                MetrikaDomainCrawlState state = metrikaCrawlStateService.getDomainCrawlState(domain);
                Set<Long> verifiedCounters = verifiedDomainCounters.get(domain);
                try {
                    Set<Long> counters = state.getEnabledCounters().stream()
                            .filter(verifiedCounters::contains)
                            .collect(Collectors.toSet());
                    if (counters.isEmpty()) {
                        return;
                    }

                    DOMAIN.set(tableWriter, domain);
                    COUNTERS.set(tableWriter, counters);
                    tableWriter.rowEnd();
                } catch (YtException e) {
                    throw new RuntimeException(e);
                }
            });
        });

        ytService.inTransaction(exportPath).execute(cypressService -> {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(YT_TABLE_SCHEMA);
            if (cypressService.exists(exportPath)) {
                cypressService.remove(exportPath);
            }

            cypressService.create(exportPath, YtNode.NodeType.TABLE, true, attributes);
            cypressService.writeTable(exportPath, tableData);

            return true;
        });

        return Result.SUCCESS;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_METRIKA_CRAWL_SETTINGS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 45 3-23/8 * * *");
    }

    @Required
    public void setExportPath(YtPath exportPath) {
        this.exportPath = exportPath;
    }
}
