package ru.yandex.webmaster3.worker.mirrors;

import java.util.*;
import java.util.stream.Collectors;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.PeriodicTaskData;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author tsyplyaev
 */
public class ChangeMainMirrorPeriodicTask
        extends PeriodicTask<ChangeMainMirrorPeriodicTask.TaskState>
{
    private static final Logger log = LoggerFactory.getLogger(ChangeMainMirrorPeriodicTask.class);

    protected static final Duration RETRY_PERIOD = Duration.standardDays(1);

    private MirrorRequestStateService mirrorRequestStateService;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.CHANGE_MAIN_MIRROR_PERIODIC;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        state = new TaskState();
        checkMirrorRequests();
        return new Result(TaskResult.SUCCESS);
    }

    // For testing
    protected void checkMirrorRequests() {
        List<MirrorRequest> mirrorRequests = mirrorRequestStateService.getAllMirrorRequests();
        DateTime oldRequests = DateTime.now().minus(RETRY_PERIOD);

        Map<WebmasterHostId, List<MirrorRequest>> groupedRequests = mirrorRequests.stream()
                .collect(Collectors.groupingBy(MirrorRequest::getHostId));
        // reverse sort
        groupedRequests.values().forEach(requests ->
                requests.sort(Comparator.comparing(MirrorRequest::getCreateDate).reversed()));

        for (Map.Entry<WebmasterHostId, List<MirrorRequest>> entry : groupedRequests.entrySet()) {
            List<MirrorRequest> requests = entry.getValue();
            MirrorRequest latestRequest = requests.get(0);
            if (latestRequest.getState() == MirrorRequestStateEnum.NEW
                    && oldRequests.isAfter(latestRequest.getUpdateDate()))
            {
                try {
                    log.info("Recheck: {} {} {} {}", latestRequest.getHostId(), latestRequest.getRequestId(),
                            latestRequest.getState(), latestRequest.getUpdateDate());
                    mirrorRequestStateService.executeMirrorRequest(latestRequest);
                } catch (WebmasterYdbException e) {
                    log.error("Unable to recheck request: {} {} {} {}", latestRequest.getHostId(), latestRequest.getRequestId(),
                            latestRequest.getState(), latestRequest.getUpdateDate(), e);
                }
            }

            for (int i = 1; i < requests.size(); i++) {
                MirrorRequest request = requests.get(i);
                if (request.getState() == MirrorRequestStateEnum.NEW) {
                    try {
                        mirrorRequestStateService.markRequestAsFailed(request);
                    } catch (WebmasterYdbException e) {
                        log.error("Unable to fix old request: {} {} {}", request.getHostId(), request.getRequestId(),
                                request.getState(), e);
                    }
                }
            }
        }
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 15 0 * * *");
    }

    @Required
    public void setMirrorRequestStateService(MirrorRequestStateService mirrorRequestStateService) {
        this.mirrorRequestStateService = mirrorRequestStateService;
    }

    static class TaskData extends PeriodicTaskData {
    }

    static class TaskState implements PeriodicTaskState {
    }
}
