package ru.yandex.webmaster3.worker.mirrors;

import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import lombok.AllArgsConstructor;
import lombok.NonNull;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.mirrors.dao.DisplayNameCHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;


@AllArgsConstructor(onConstructor_ = @Autowired)
public class ImportDisplayNamesTask extends AbstractImportJupiterExportTask {

    private static final int LINES_COUNT = 1024;

    private static final Map<String, String> SOURCE_EXPRESSIONS = new ImmutableMap.Builder<String, String>()
            .put(DisplayNameCHDao.F.HOST, "$url2HostId(Host)")
            .put(DisplayNameCHDao.F.DISPLAY_NAME, "Url::GetHost(`BeautyHost`)")
            .build();
    private static final String SUFFIX_TO_BEAUTY_HOSTS_TABLE = "/webmaster/beauty_host_names";

    @Override
    // если увеличивать то стоит переписать yql в prepareIntermediateTable
    protected int getShardsCount() {
        return 1;
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        final String dateString = IN_YQL_QUERY_DATE_FORMATTER.print(imprt.getDateTo());
        final YtPath sourceTable = imprt.getSourceTable();
        final String fields = getTable().getFields().stream()
                .map(CHField::getName)
                .filter(chField -> !chField.equals(DisplayNameCHDao.F.DATE))
                .map(SOURCE_EXPRESSIONS::get)
                .collect(Collectors.joining(" || '\\t' || ", "('" + dateString + "'|| '\\t' || ", " || '\\n')"));


        return YqlQueryBuilder.newBuilder()
                .cluster(sourceTable)
                .appendFDefinition(YqlFunctions.URL_2_HOST_ID)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT 0 as ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data " +
                        "FROM (\n")
                .appendText("SELECT")
                .appendText("((Digest::Fnv64($url2HostId(`Host`))) % " + LINES_COUNT + ") as RowId,")
                .appendText(fields).appendText("as data ")
                .appendText("FROM")
                .appendTable(sourceTable)
                .appendText(" ) \n GROUP BY RowId;")
                .appendText("COMMIT;\n\n");
    }

    @Override
    protected CHTable getTable() {
        return DisplayNameCHDao.TABLE;
    }

    @Override
    protected String getJupiterSuffix() {
        return SUFFIX_TO_BEAUTY_HOSTS_TABLE;
    }

    @Override
    protected @NonNull String getBaseState() {
        return jupiterUtils.getCurrentState();
    }

    @Override
    protected TableType getTableType() {
        return TableType.DISPLAY_NAME;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.DISPLAY_NAME;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0/3 * * * *");
    }
}
