package ru.yandex.webmaster3.worker.mirrors;

import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.mirrors.dao.MainMirrorsCHDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHField;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.yql.YqlFunctions;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType.MIRRORS;

@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ImportMirrorsTask extends AbstractImportJupiterExportTask {

    private static final int LINES_COUNT = 512;

    private final static Duration UPDATE_BASE_HOURS = Duration.standardHours(6);

    private static final Map<String, String> SOURCE_EXPRESSIONS = new ImmutableMap.Builder<String, String>()
            .put(MainMirrorsCHDao.F.HOST, "$url2HostId(`Host`)")
            .put(MainMirrorsCHDao.F.MAIN_HOST, "$url2HostId(`MainHost`)")
            .build();
    private static final String SUFFIX_TO_MIRRORS_TABLE = "/mirrors/mirrors";

    private final SettingsService settingsService;

    @Value("${webmaster3.worker.uploadWebmasterHostsTask.arnold.export.path}")
    public YtPath allHostsWebmasterTable;

    @Override
    // если увеличивать то стоит переписать yql в prepareIntermediateTable
    protected int getShardsCount() {
        return 1;
    }

    @Override
    // проверяем соответствие стейта базы в продакшене и у нас, а также проверяем версию таблицы
    // если стейты идентичны, а версия отличается менее чем на 6 часов, то импорт не делаем
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad latestImport) throws Exception {
        return ytService.inTransaction(tablePath).query(cypressService -> {
            String baseState = getBaseState();

            final YtPath priemkaExportTablePath = getPathToTableInExport(baseState, getJupiterSuffix());

            final YtNode node = cypressService.getNode(priemkaExportTablePath);
            final String data = Objects.requireNonNullElse(latestImport.getData(), "0");


            CommonDataType commonDataType = CommonDataType.valueOf(getType().name());
            var lastBaseState = settingsService.getSettingUncached(commonDataType).getValue();
            var lastImportedBaseVersion = new Instant(Long.parseLong(data));
            baseState = JupiterUtils.getStateWithOnlyDigits(baseState);

            if (!baseState.equals(lastBaseState)
                    || lastImportedBaseVersion.plus(UPDATE_BASE_HOURS).isBefore(Instant.now())) {
                final LocalDate date = node.getUpdateTime().toLocalDate();
                settingsService.update(commonDataType, baseState);

                log.info("previously base state -- {}, current base state -- {} previously base version -- {}",
                        lastBaseState, baseState, lastImportedBaseVersion);

                return latestImport.withData(String.valueOf(Instant.now().getMillis()))
                        .withSourceTable(priemkaExportTablePath, date, date);
            }
            // ничего нового
            log.info("base state -- {} and base version -- {} is actual", lastBaseState, lastImportedBaseVersion);
            return latestImport.withState(YtClickhouseDataLoadState.DONE);
        });
    }

    @Override
    // импортим главные зеркала для всех хостов вебмастера, которые сами не являются главным зеркалом
    protected YqlQueryBuilder prepareIntermediateTable(final YtClickhouseDataLoad imprt) {
        final String dateString = IN_YQL_QUERY_DATE_FORMATTER.print(imprt.getDateTo());
        final YtPath sourceTable = imprt.getSourceTable();
        final String fields = getTable().getFields().stream()
                .map(CHField::getName)
                .filter(chField -> !chField.equals(MainMirrorsCHDao.F.DATE))
                .map(SOURCE_EXPRESSIONS::get)
                .collect(Collectors.joining(" || '\\t' || ", "('" + dateString + "'|| '\\t' || ", " || '\\n')"));

        final Map<String, String> values = Map.of(
                "jupiterMirrors", sourceTable.toYqlPath(),
                "allHostsWebmasterTable", allHostsWebmasterTable.toYqlPath(),
                "INTERMEDIATE_TABLE", INTERMEDIATE_TABLE,
                "LINES_COUNT", String.valueOf(LINES_COUNT),
                "fields", fields
        );
        StrSubstitutor sub = new StrSubstitutor(values);

        String template = "PRAGMA yt.MaxRowWeight = '128M';\n" +
                "PRAGMA yt.InferSchema = '1';\n" +
                "INSERT INTO ${INTERMEDIATE_TABLE} " +
                "SELECT  0 as ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM (\n" +
                "SELECT ((Digest::Fnv64($url2HostId(`Host`))) % ${LINES_COUNT} ) as RowId, ${fields} as data " +
                "FROM ${jupiterMirrors} as jupiterMirrors " +
                "INNER JOIN (SELECT DISTINCT Host AS host_url FROM ${allHostsWebmasterTable} ) AS allHostsTable " +
                "ON jupiterMirrors.Host == allHostsTable.host_url  WHERE `Host` != `MainHost`) \n" +
                "GROUP BY RowId;\n COMMIT;\n\n";

        String query = sub.replace(template);

        return YqlQueryBuilder.newBuilder()
                .cluster(sourceTable)
                .appendFDefinition(YqlFunctions.URL_2_HOST_ID)
                .appendText(query);

    }

    @Override
    protected CHTable getTable() {
        return MainMirrorsCHDao.TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.MIRRORS;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return MIRRORS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0/3 * * * *");
    }

    @Override
    protected String getJupiterSuffix() {
        return SUFFIX_TO_MIRRORS_TABLE;
    }

    @Override
    protected @NonNull String getBaseState() {
        return jupiterUtils.getCurrentState();
    }
}
