package ru.yandex.webmaster3.worker.mirrors;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.mirrors.dao.MainMirrorRequestsYDao;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;
import ru.yandex.webmaster3.storage.mirrors.service.MainMirrorService;

/**
 * @author aherman
 */
public class MirrorRequestStateService {
    private static final Logger log = LoggerFactory.getLogger(MirrorRequestStateService.class);
    private static final RequestConfig REQUEST_CONFIG = RequestConfig.custom()
            .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
            .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
            .setSocketTimeout(5 * 60 * 1000)
            .build();
    private static final int MAX_ATTEMPTS_COUNT = 3;

    @Nullable private final CloseableHttpClient httpClient;
    @Nullable private final String robotMirrorUrl;

    private final MainMirrorRequestsYDao mainMirrorRequestsYDao;
    private final DoMirrorRequest doRequest;

    @Autowired
    public MirrorRequestStateService(MainMirrorRequestsYDao mainMirrorRequestsYDao, String robotMirrorUrl) {
        this.robotMirrorUrl = Preconditions.checkNotNull(robotMirrorUrl);
        this.mainMirrorRequestsYDao = Preconditions.checkNotNull(mainMirrorRequestsYDao);
        this.httpClient = HttpClientBuilder.create()
                .setDefaultRequestConfig(REQUEST_CONFIG)
                .setConnectionTimeToLive(10, TimeUnit.MINUTES)
                .build();
        this.doRequest = this::doMirrorRequest;
    }

    private MirrorServiceResponse doMirrorRequest(MirrorServiceRequest req) throws IOException {
        Preconditions.checkNotNull(robotMirrorUrl);
        Preconditions.checkNotNull(httpClient);
        log.debug("Sending request: {}", req);
        HttpGet get = new HttpGet(req.toUriString(robotMirrorUrl));
        HttpResponse httpResponse = this.httpClient.execute(get);
        log.info("Mirrors robot response status code: {}", httpResponse.getStatusLine().getStatusCode());
        HttpEntity entity = httpResponse.getEntity();
        String jsonResponse = EntityUtils.toString(entity);
        log.info("Got json response: {}", jsonResponse);
        EntityUtils.consume(entity);
        return MirrorServiceResponse.fromJson(jsonResponse);
    }

    @VisibleForTesting
    MirrorRequestStateService(MainMirrorRequestsYDao mainMirrorRequestsYDao,
                              DoMirrorRequest doRequest) {
        this.mainMirrorRequestsYDao = Preconditions.checkNotNull(mainMirrorRequestsYDao);
        this.doRequest = Preconditions.checkNotNull(doRequest);
        this.httpClient = null;
        this.robotMirrorUrl = null;
    }

    public void destroy() {
        IOUtils.closeQuietly(httpClient);
    }

    @VisibleForTesting
    interface DoMirrorRequest {
        MirrorServiceResponse get(MirrorServiceRequest req) throws Exception;
    }

    public List<MirrorRequest> getAllMirrorRequests() {
        try {
            return mainMirrorRequestsYDao.listAllRequests();
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to read main mirror requests",
                    new WebmasterErrorResponse.YDBErrorResponse(this.getClass(), e), e);
        }
    }

    public boolean executeMirrorRequest(MirrorRequest mirrorRequest)  {
        if (mirrorRequest.getState() != MirrorRequestStateEnum.NEW) {
            log.info("Request not in {} state, skip: {} {} {}", MirrorRequestStateEnum.NEW,
                    mirrorRequest.getRequestId(),
                    mirrorRequest.getHostId(),
                    mirrorRequest.getState()
            );
            return true;
        }

        log.info("Change host mirror: {} {} {} -> {} {} {}",
                mirrorRequest.getRequestId(),
                mirrorRequest.getHostId(),
                mirrorRequest.getOldMainMirrorHostId(),
                mirrorRequest.getNewMainMirrorHostId(),
                mirrorRequest.getState(),
                mirrorRequest.getUpdateDate()
        );


        MirrorServiceResponse response;
        try {
            response = doRequest.get(MirrorServiceRequest.fromMirrorRequest(mirrorRequest));
        } catch (Exception x) {
            log.error("Unable to change host mirror: {} {} {} -> {}",
                      mirrorRequest.getRequestId(), mirrorRequest.getHostId(),
                      mirrorRequest.getOldMainMirrorHostId(), mirrorRequest.getNewMainMirrorHostId(), x);
            mirrorRequest = updateAttemptsCountAndState(mirrorRequest, mirrorRequest.getAttemptCount() + 1);
            mainMirrorRequestsYDao.saveRequest(mirrorRequest);
            return false;
        }

        String responseJson;
        try {
            responseJson = response.asJson();
        } catch (Exception e) {
            responseJson = "{\"error\":\"Unable to save Mirror response\"}";
        }

        mirrorRequest = mirrorRequest.withServiceResponse(responseJson);
        mirrorRequest = updateState(mirrorRequest, response);
        mainMirrorRequestsYDao.saveRequest(mirrorRequest);

        log.info("Mirror request processed: {} {} {} -> {} {}",
                mirrorRequest.getRequestId(),
                mirrorRequest.getHostId(),
                mirrorRequest.getOldMainMirrorHostId(),
                mirrorRequest.getNewMainMirrorHostId(),
                mirrorRequest.getState()
        );
        return mirrorRequest.getState() != MirrorRequestStateEnum.SERVICE_ERROR && mirrorRequest.getState() != MirrorRequestStateEnum.INTERNAL_ERROR;
    }

    private static MirrorRequest updateState(MirrorRequest mirrorRequest, MirrorServiceResponse response) {
        switch (response.getStatus()) {
            case OK:
                mirrorRequest = mirrorRequest.withState(MirrorRequestStateEnum.WAITING);
                break;

            case ERROR_USER:
                mirrorRequest = mirrorRequest.withState(MirrorRequestStateEnum.USER_ERROR);
                break;

            case ERROR_INTERNAL:
                mirrorRequest = mirrorRequest.withState(MirrorRequestStateEnum.SERVICE_ERROR);
                break;

            case ERROR_TIMEOUT:
                mirrorRequest = mirrorRequest.withState(MirrorRequestStateEnum.SERVICE_ERROR);
                break;

            default:
                mirrorRequest = mirrorRequest.withState(MirrorRequestStateEnum.INTERNAL_ERROR);
        }

        return mirrorRequest;
    }

    public void markRequestAsFailed(MirrorRequest request)  {
        log.info("Mark request as INTERNAL_ERROR: {} {} {} {}",
                request.getRequestId(),
                request.getHostId(),
                request.getState(),
                request.getUpdateDate()
        );
        request = request.withState(MirrorRequestStateEnum.INTERNAL_ERROR);
        mainMirrorRequestsYDao.saveRequest(request);
    }

    public void updateWaitingRequest(WebmasterHostId currentMirrorHostId, MirrorRequest latestRequest)
             {
        if (latestRequest == null) {
            return;
        }
        if (latestRequest.getState() != MirrorRequestStateEnum.WAITING) {
            return;
        }

        DateTime oldRequestDate = DateTime.now().minus(MainMirrorService.EXPIRE_PERIOD);
        if (currentMirrorHostId.equals(latestRequest.getNewMainMirrorHostId())) {
            log.info("Completing mirror request for host {}", latestRequest.getHostId());
            latestRequest = latestRequest.withState(MirrorRequestStateEnum.SUCCESS);
            mainMirrorRequestsYDao.saveRequest(latestRequest);
        } else if (oldRequestDate.isAfter(latestRequest.getUpdateDate())) {
            log.info("Expiring mirror request for host {}", latestRequest.getHostId());
            latestRequest = latestRequest.withState(MirrorRequestStateEnum.EXPIRED);
            mainMirrorRequestsYDao.saveRequest(latestRequest);
        }
    }

    private static MirrorRequest updateAttemptsCountAndState(MirrorRequest mr, int attemptsCount) {
        MirrorRequestStateEnum state = mr.getState();
        if (attemptsCount > MAX_ATTEMPTS_COUNT) {
            state = MirrorRequestStateEnum.INTERNAL_ERROR;
        }
        return mr
                .withState(state)
                .withAttempts(attemptsCount);
    }
}
